<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreUnitRequest;
use App\Http\Requests\SocietyAdmin\UpdateUnitRequest;
use App\Models\Building;
use App\Models\Society;
use App\Models\Unit;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class UnitController extends Controller
{
    /**
     * Display a listing of units.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        // Ensure user belongs to this society
        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $query = Unit::where('society_id', $society->id)
            ->with(['building:id,name', 'residents:id,name,phone,unit_id', 'maintenanceStructure:id,name']);

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('unit_no', 'like', "%{$search}%");
            });
        }

        // Filter by building
        if ($request->has('building_id') && $request->get('building_id') !== 'all') {
            $query->where('building_id', $request->get('building_id'));
        }

        // Filter by status
        if ($request->has('status') && $request->get('status') !== 'all') {
            $query->where('status', $request->get('status'));
        }

        // Filter by type
        if ($request->has('type') && $request->get('type') !== 'all') {
            $query->where('type', $request->get('type'));
        }

        $units = $query->latest()->paginate(15)->withQueryString();

        // Get buildings for filter
        $buildings = Building::where('society_id', $society->id)
            ->orderBy('name')
            ->get(['id', 'name']);

        // Get maintenance structures
        $maintenanceStructures = \App\Models\MaintenanceStructure::where('society_id', $society->id)
            ->orderBy('name')
            ->get(['id', 'name']);

        // Calculate Stats
        $totalUnits = Unit::where('society_id', $society->id)->count();
        $occupiedUnits = Unit::where('society_id', $society->id)->where('status', 'occupied')->count();
        $vacantUnits = Unit::where('society_id', $society->id)->where('status', 'vacant')->count();

        return Inertia::render('SocietyAdmin/Units/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'units' => $units,
            'buildings' => $buildings,
            'maintenanceStructures' => $maintenanceStructures,

            'filters' => $request->only(['search', 'building_id', 'status', 'type']),
            'stats' => [
                'total_units' => $totalUnits,
                'occupied_units' => $occupiedUnits,
                'vacant_units' => $vacantUnits,
            ],
        ]);
    }

    /**
     * Show the form for creating a new unit.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $buildings = Building::where('society_id', $society->id)
            ->orderBy('name')
            ->get(['id', 'name']);

        return Inertia::render('SocietyAdmin/Units/Create', [
            'society' => $society->only(['id', 'name', 'code']),
            'buildings' => $buildings,
        ]);
    }

    /**
     * Store a newly created unit.
     */
    public function store(StoreUnitRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;

        if (isset($data['building_id']) && $data['building_id'] === 'none') {
            $data['building_id'] = null;
        }

        // New units cannot have residents, so they cannot be occupied or rented
        if (in_array($data['status'], ['occupied', 'rented'])) {
            $data['status'] = 'vacant';
        }

        $unit = Unit::create($data);

        return redirect()->route('society.units.index', $society)
            ->with('success', 'Unit created successfully.');
    }

    /**
     * Display the specified unit.
     */
    public function show(Society $society, Unit $unit): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $unit->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: View only, no show page
        if ($user->hasRole('sub_admin') && ! $user->hasPermission('units.view')) {
            abort(403, 'You do not have permission to view unit details.');
        }

        $unit->load(['building', 'user:id,name,phone,email', 'society:id,name']);

        return Inertia::render('SocietyAdmin/Units/Show', [
            'society' => $society->only(['id', 'name', 'code']),
            'unit' => $unit,
        ]);
    }

    /**
     * Show the form for editing the specified unit.
     */
    public function edit(Society $society, Unit $unit): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $unit->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No edit permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view and create units.');
        }

        $unit->load(['building', 'user:id,name']);

        $buildings = Building::where('society_id', $society->id)
            ->orderBy('name')
            ->get(['id', 'name']);



        return Inertia::render('SocietyAdmin/Units/Edit', [
            'society' => $society->only(['id', 'name', 'code']),
            'unit' => $unit,
            'buildings' => $buildings,

        ]);
    }

    /**
     * Update the specified unit.
     */
    public function update(UpdateUnitRequest $request, Society $society, Unit $unit): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $unit->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No update permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view and create units.');
        }

        $data = $request->validated();

        // Enforce status consistency based on resident assignment
        if ($unit->user_id) {
            // If resident exists, unit cannot be vacant
            if ($data['status'] === 'vacant') {
                $data['status'] = 'occupied';
            }
        } else {
            // If no resident exists, unit cannot be occupied or rented
            if (in_array($data['status'], ['occupied', 'rented'])) {
                $data['status'] = 'vacant';
            }
        }

        $unit->update($data);

        return redirect()->route('society.units.index', $society)
            ->with('success', 'Unit updated successfully.');
    }

    /**
     * Remove the specified unit.
     */
    public function destroy(Society $society, Unit $unit): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $unit->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No delete permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view and create units.');
        }

        // Free up user's unit assignment
        if ($unit->user_id) {
            User::where('id', $unit->user_id)->update(['unit_id' => null]);
        }

        $unit->delete();

        return redirect()->route('society.units.index', $society)
            ->with('success', 'Unit deleted successfully.');
    }
}
