<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Inertia\Response;

class TeamController extends Controller
{
    /**
     * Get allowed roles for society team members
     */
    private function getAllowedRoles(): array
    {
        return ['sub_admin', 'accountant', 'collector', 'guard', 'staff', 'vendor'];
    }

    /**
     * Display a listing of team members.
     */
    public function index(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $allowedRoleNames = $this->getAllowedRoles();

        $team = User::where('society_id', $society->id)
            ->whereHas('role', function ($q) use ($allowedRoleNames) {
                $q->whereIn('name', $allowedRoleNames);
            })
            ->with('role:id,name,display_name')
            ->orderBy('name')
            ->get();

        $roles = Role::whereIn('name', $allowedRoleNames)
            ->select('id', 'name', 'display_name')
            ->orderBy('display_name')
            ->get();

        // Stats
        $stats = [
            'total_members' => $team->count(),
            'active_members' => $team->where('status', 'active')->count(),
            'by_role' => $team->groupBy('role.name')->map->count(),
        ];

        return Inertia::render('SocietyAdmin/Team/Index', [
            'society' => [
                'id' => $society->id,
                'name' => $society->name,
            ],
            'team' => $team,
            'roles' => $roles,
            'stats' => $stats,
        ]);
    }

    /**
     * Store a newly created team member.
     */
    public function store(Request $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $allowedRoleNames = $this->getAllowedRoles();

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255', 'unique:users,email'],
            'phone' => ['required', 'string', 'max:20', 'unique:users,phone'],
            'password' => ['required', 'string', 'min:6', 'confirmed'],
            'role_id' => [
                'required',
                'exists:roles,id',
                function ($attribute, $value, $fail) use ($allowedRoleNames) {
                    $role = Role::find($value);
                    if (!$role || !in_array($role->name, $allowedRoleNames)) {
                        $fail('Invalid role selected.');
                    }
                },
            ],
            'status' => ['required', 'in:active,inactive,suspended'],
        ]);

        User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'password' => Hash::make($validated['password']),
            'role_id' => $validated['role_id'],
            'society_id' => $society->id,
            'status' => $validated['status'],
            'directory_visible' => false,
        ]);

        return redirect()->route('society.team.index', $society)
            ->with('success', 'Team member added successfully.');
    }

    /**
     * Update the specified team member.
     */
    public function update(Request $request, Society $society, User $team): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $team->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $allowedRoleNames = $this->getAllowedRoles();

        // Ensure the user being updated is a team member (not a resident)
        if (!in_array($team->role->name, $allowedRoleNames)) {
            abort(403, 'This user is not a team member.');
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['nullable', 'email', 'max:255', Rule::unique('users', 'email')->ignore($team->id)],
            'phone' => ['required', 'string', 'max:20', Rule::unique('users', 'phone')->ignore($team->id)],
            'password' => ['nullable', 'string', 'min:6', 'confirmed'],
            'role_id' => [
                'required',
                'exists:roles,id',
                function ($attribute, $value, $fail) use ($allowedRoleNames) {
                    $role = Role::find($value);
                    if (!$role || !in_array($role->name, $allowedRoleNames)) {
                        $fail('Invalid role selected.');
                    }
                },
            ],
            'status' => ['required', 'in:active,inactive,suspended'],
        ]);

        $team->update([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'phone' => $validated['phone'],
            'role_id' => $validated['role_id'],
            'status' => $validated['status'],
        ]);

        if (!empty($validated['password'])) {
            $team->update(['password' => Hash::make($validated['password'])]);
        }

        return redirect()->route('society.team.index', $society)
            ->with('success', 'Team member updated successfully.');
    }

    /**
     * Remove the specified team member.
     */
    public function destroy(Society $society, User $team): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $team->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $allowedRoleNames = $this->getAllowedRoles();

        // Ensure the user being deleted is a team member (not a resident)
        if (!in_array($team->role->name, $allowedRoleNames)) {
            abort(403, 'This user is not a team member.');
        }

        // Prevent self-deletion
        if ($team->id === $user->id) {
            return redirect()->route('society.team.index', $society)
                ->with('error', 'You cannot delete yourself.');
        }

        $team->delete();

        return redirect()->route('society.team.index', $society)
            ->with('success', 'Team member removed successfully.');
    }
}
