<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\UpdateSocietySettingsRequest;
use App\Models\Society;
use App\Models\SocietySetting;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class SettingsController extends Controller
{
    /**
     * Display the society settings page.
     */
    public function index(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $settings = SocietySetting::firstOrCreate(
            ['society_id' => $society->id],
            []
        );

        return Inertia::render('SocietyAdmin/Settings/Index', [
            'society' => $society,
            'settings' => $settings,
        ]);
    }

    /**
     * Update the society settings.
     */
    public function update(UpdateSocietySettingsRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $settings = SocietySetting::firstOrCreate(
            ['society_id' => $society->id],
            []
        );

        $data = $request->validated();
        
        // Update Society Details
        $societyData = collect($data)->only([
            'name', 'code', 'registration_no', 'address', 'city', 'state', 'pincode', 
            'phone', 'email', 'gst_no', 'pan_no', 'rules'
        ])->toArray();
        
        $society->update($societyData);

        // Update Settings
        // Remove society fields from data to avoid mass assignment errors on settings model if it was strict, 
        // though `fillable` protects it. But let's be clean.
        $settingsData = collect($data)->except([
            'name', 'code', 'registration_no', 'address', 'city', 'state', 'pincode', 
            'phone', 'email', 'gst_no', 'pan_no', 'rules'
        ])->toArray();
        
        // Convert "none" values to null for optional fields
        if (isset($settingsData['smtp_encryption']) && $settingsData['smtp_encryption'] === 'none') {
            $settingsData['smtp_encryption'] = null;
        }

        // Handle boolean fields that might come as strings
        $booleanFields = [
            'razorpay_enabled', 'stripe_enabled', 'payu_enabled', 'cashfree_enabled',
            'email_enabled', 'sms_enabled', 'whatsapp_enabled', 'fcm_enabled',
            'notify_bill_generated', 'notify_payment_received', 'notify_visitor_arrival',
            'notify_maintenance_request', 'notify_announcement',
            'enable_visitor_management', 'enable_maintenance_tickets', 'enable_complaints',
            'enable_events', 'enable_documents', 'enable_directory', 'enable_notices',
            'enable_polls', 'enable_marketplace', 'enable_facility_booking',
            'require_otp_for_login', 'require_otp_for_payment', 'enable_2fa',
        ];

        foreach ($booleanFields as $field) {
            if (isset($settingsData[$field])) {
                $settingsData[$field] = filter_var($settingsData[$field], FILTER_VALIDATE_BOOLEAN);
            }
        }

        $settings->update($settingsData);

        return redirect()->route('society.settings.index', $society)
            ->with('success', 'Settings updated successfully.');
    }
}
