<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Models\Society;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class RevenueController extends Controller
{
    /**
     * Display revenue dashboard.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Filters
        $dateFrom = $request->get('date_from', now()->startOfMonth()->toDateString());
        $dateTo = $request->get('date_to', now()->endOfMonth()->toDateString());

        // Base Query
        $query = Transaction::where('society_id', $society->id)
            ->whereBetween('transaction_date', [$dateFrom, $dateTo]);

        // Aggregates
        $totalIncome = (clone $query)->credits()->sum('amount');
        $totalExpense = (clone $query)->debits()->sum('amount');
        $netRevenue = $totalIncome - $totalExpense;

        // Breakdown by Category
        $incomebyCategory = (clone $query)->credits()
            ->selectRaw('category, sum(amount) as total')
            ->groupBy('category')
            ->pluck('total', 'category');

        $expensebyCategory = (clone $query)->debits()
            ->selectRaw('category, sum(amount) as total')
            ->groupBy('category')
            ->pluck('total', 'category');

        // Recent Transactions
        $recentTransactions = Transaction::where('society_id', $society->id)
            ->with(['bankAccount:id,account_name', 'createdBy:id,name', 'reference'])
            ->latest('transaction_date')
            ->latest('id')
            ->limit(10)
            ->get()
            ->map(function ($tx) {
                return [
                    'id' => $tx->id,
                    'type' => $tx->type,
                    'amount' => (float) $tx->amount,
                    'date' => $tx->transaction_date->format('Y-m-d'),
                    'description' => $tx->description,
                    'reference' => $tx->formatted_reference,
                    'account_name' => $tx->bankAccount->account_name,
                    'balance_after' => (float) $tx->balance_after,
                ];
            });

        return Inertia::render('SocietyAdmin/Revenue/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'filters' => [
                'date_from' => $dateFrom,
                'date_to' => $dateTo,
            ],
            'stats' => [
                'total_income' => $totalIncome,
                'total_expense' => $totalExpense,
                'net_revenue' => $netRevenue,
            ],
            'charts' => [
                'income_by_category' => $incomebyCategory,
                'expense_by_category' => $expensebyCategory,
            ],
            'recent_transactions' => $recentTransactions,
        ]);
    }

    /**
     * Display transaction history list.
     */
    public function history(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $query = Transaction::where('society_id', $society->id)
            ->with(['bankAccount:id,account_name', 'createdBy:id,name', 'reference']);

        // Filters
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('description', 'like', "%{$search}%")
                    ->orWhere('payment_method', 'like', "%{$search}%")
                    ->orWhere('category', 'like', "%{$search}%");
            });
        }

        if ($request->has('type')) {
            $query->where('type', $request->get('type'));
        }

        if ($request->has('date_from')) {
            $query->where('transaction_date', '>=', $request->get('date_from'));
        }

        if ($request->has('date_to')) {
            $query->where('transaction_date', '<=', $request->get('date_to'));
        }

        if ($request->has('bank_account_id')) {
            $query->where('bank_account_id', $request->get('bank_account_id'));
        }

        $transactions = $query->latest('transaction_date')
            ->latest('id')
            ->paginate(15)
            ->withQueryString()
            ->through(function ($tx) {
                return [
                    'id' => $tx->id,
                    'type' => $tx->type,
                    'date' => $tx->transaction_date->format('Y-m-d H:i:s'),
                    'description' => $tx->description,
                    'reference' => $tx->formatted_reference,
                    'payment_method' => $tx->payment_method ?? 'N/A',
                    'debit' => $tx->type === 'debit' ? (float) $tx->amount : 0,
                    'credit' => $tx->type === 'credit' ? (float) $tx->amount : 0,
                    'balance' => (float) $tx->balance_after,
                    'bank_account' => $tx->bankAccount ? $tx->bankAccount->account_name : 'Unknown',
                    'category' => ucwords(str_replace('_', ' ', $tx->category)),
                    'created_by_name' => $tx->createdBy ? $tx->createdBy->name : 'System',
                ];
            });

        $bankAccounts = \App\Models\BankAccount::where('society_id', $society->id)
            ->select('id', 'account_name')
            ->get();

        return Inertia::render('SocietyAdmin/Revenue/History', [
            'society' => $society->only(['id', 'name', 'code']),
            'transactions' => $transactions,
            'bankAccounts' => $bankAccounts,
            'filters' => $request->only(['search', 'type', 'date_from', 'date_to', 'bank_account_id']),
        ]);
    }
}
