<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreResidentRequest;
use App\Http\Requests\SocietyAdmin\UpdateResidentRequest;
use App\Models\Role;
use App\Models\Society;
use App\Models\Unit;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;
use Inertia\Response;

class ResidentController extends Controller
{
    /**
     * Display a listing of residents.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        // Ensure user belongs to this society
        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $residentRole = Role::where('name', 'resident')->first();

        if (! $residentRole) {
            abort(500, 'Resident role not found. Please seed roles first.');
        }

        $query = User::where('society_id', $society->id)
            ->where('role_id', $residentRole->id)
            ->with(['unit:id,unit_no,building_id', 'unit.building:id,name'])
            ->select(['id', 'name', 'email', 'phone', 'status', 'directory_visible', 'unit_id', 'type', 'parent_id', 'created_at']);

        // Search
        if ($request->has('search')) {
            $query->search($request->get('search'));
        }

        // Filter by status
        if ($request->has('status') && $request->get('status') !== 'all') {
            $query->where('status', $request->get('status'));
        }

        // Filter by unit
        if ($request->has('unit_id') && $request->get('unit_id') !== 'all') {
            $query->where('unit_id', $request->get('unit_id'));
        }

        // Filter by directory visibility
        if ($request->has('directory_visible') && $request->get('directory_visible') !== 'all') {
            $query->where('directory_visible', $request->boolean('directory_visible'));
        }

        // Filter by type
        if ($request->has('type') && $request->get('type') !== 'all') {
            $query->where('type', $request->get('type'));
        }

        $residents = $query->latest()->paginate(15)->withQueryString();

        // Get available units for filter
        // Get all units for the modal
        $units = Unit::where('society_id', $society->id)
            ->with('building:id,name')
            ->orderBy('unit_no')
            ->get();

        // Get all buildings for the modal
        $buildings = \App\Models\Building::where('society_id', $society->id)->get(['id', 'name']);

        // Calculate Stats by Type
        $baseQuery = User::where('society_id', $society->id)->where('role_id', $residentRole->id);
        $totalMembers = (clone $baseQuery)->count();
        $activeMembers = (clone $baseQuery)->where('status', 'active')->count();
        $ownerCount = (clone $baseQuery)->where('type', 'owner')->count();
        $tenantCount = (clone $baseQuery)->where('type', 'tenant')->count();
        $familyCount = (clone $baseQuery)->where('type', 'family_member')->count();

        return Inertia::render('SocietyAdmin/Residents/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'residents' => $residents,
            'units' => $units,
            'buildings' => $buildings,
            'filters' => $request->only(['search', 'status', 'unit_id', 'directory_visible', 'type']),
            'stats' => [
                'total_members' => $totalMembers,
                'active_members' => $activeMembers,
                'owners' => $ownerCount,
                'tenants' => $tenantCount,
                'family_members' => $familyCount,
            ],
        ]);
    }

    /**
     * Show the form for creating a new resident.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Get available units (vacant or can be reassigned)
        $units = Unit::where('society_id', $society->id)
            ->with('building:id,name')
            ->orderBy('unit_no')
            ->get();

        return Inertia::render('SocietyAdmin/Residents/Create', [
            'society' => $society->only(['id', 'name', 'code']),
            'units' => $units,
        ]);
    }

    /**
     * Store a newly created resident.
     */
    public function store(StoreResidentRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $residentRole = Role::where('name', 'resident')->first();

        if (! $residentRole) {
            return redirect()->back()
                ->withErrors(['role' => 'Resident role not found.'])
                ->withInput();
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;
        $data['role_id'] = $residentRole->id;
        $data['password'] = Hash::make($data['password']);

        // Convert "none" values to null for optional fields
        if (isset($data['unit_id']) && $data['unit_id'] === 'none') {
            $data['unit_id'] = null;
        }

        $resident = User::create($data);

        // Update unit status if unit is assigned
        if ($resident->unit_id) {
            Unit::where('id', $resident->unit_id)
                ->update([
                    'status' => 'occupied',
                    'user_id' => $resident->id,
                ]);
        }

        return redirect()->route('society.residents.index', $society)
            ->with('success', 'Resident created successfully.');
    }

    /**
     * Display the specified resident.
     */
    public function show(Society $society, User $resident): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $resident->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: View only, no show page
        if ($user->hasRole('sub_admin') && ! $user->hasPermission('residents.view')) {
            abort(403, 'You do not have permission to view resident details.');
        }

        $resident->load(['unit.building', 'role', 'bills' => function ($q) {
            $q->latest()->limit(10);
        }, 'maintenanceTickets' => function ($q) {
            $q->latest()->limit(10);
        }, 'payments' => function ($q) {
            $q->latest()->limit(10);
        }]);

        return Inertia::render('SocietyAdmin/Residents/Show', [
            'society' => $society->only(['id', 'name', 'code']),
            'resident' => $resident,
        ]);
    }

    /**
     * Show the form for editing the specified resident.
     */
    public function edit(Society $society, User $resident): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $resident->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No edit permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view and create residents.');
        }

        $resident->load(['unit.building', 'role']);

        // Get available units
        $units = Unit::where('society_id', $society->id)
            ->with('building:id,name')
            ->orderBy('unit_no')
            ->get();

        return Inertia::render('SocietyAdmin/Residents/Edit', [
            'society' => $society->only(['id', 'name', 'code']),
            'resident' => $resident,
            'units' => $units,
        ]);
    }

    /**
     * Update the specified resident.
     */
    public function update(UpdateResidentRequest $request, Society $society, User $resident): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $resident->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No update permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view and create residents.');
        }

        $data = $request->validated();

        // Convert "none" values to null for optional fields
        if (isset($data['unit_id']) && $data['unit_id'] === 'none') {
            $data['unit_id'] = null;
        }

        // Handle password update
        if (empty($data['password'])) {
            unset($data['password']);
        } else {
            $data['password'] = Hash::make($data['password']);
        }

        $oldUnitId = $resident->unit_id;
        $newUnitId = $data['unit_id'] ?? null;

        $resident->update($data);

        // Update unit statuses
        if ($oldUnitId !== $newUnitId) {
            // Free up old unit
            if ($oldUnitId) {
                Unit::where('id', $oldUnitId)->update([
                    'status' => 'vacant',
                    'user_id' => null,
                ]);
            }

            // Assign new unit
            if ($newUnitId) {
                Unit::where('id', $newUnitId)->update([
                    'status' => 'occupied',
                    'user_id' => $resident->id,
                ]);
            }
        }

        return redirect()->route('society.residents.index', $society)
            ->with('success', 'Resident updated successfully.');
    }

    /**
     * Remove the specified resident.
     */
    public function destroy(Society $society, User $resident): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $resident->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No delete permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view and create residents.');
        }

        // Free up unit
        if ($resident->unit_id) {
            Unit::where('id', $resident->unit_id)->update([
                'status' => 'vacant',
                'user_id' => null,
            ]);
        }

        $resident->delete();

        return redirect()->route('society.residents.index', $society)
            ->with('success', 'Resident deleted successfully.');
    }
}
