<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Models\Bill;
use App\Models\Expense;
use App\Models\Payment;
use App\Models\Society;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class ReportController extends Controller
{
    /**
     * Display the reports index page.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $reportType = $request->get('type', 'collection');

        return Inertia::render('SocietyAdmin/Reports/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'reportType' => $reportType,
        ]);
    }

    /**
     * Get collection summary report.
     */
    public function collectionSummary(Request $request, Society $society)
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        // Total collections in period
        $totalCollections = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
            ->sum('amount');

        // Collections by payment method
        $byMethod = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
            ->get()
            ->groupBy('payment_method')
            ->map(function ($payments, $method) {
                return [
                    'method' => $method,
                    'amount' => (float) $payments->sum('amount'),
                    'count' => $payments->count(),
                ];
            })
            ->values();

        // Daily breakdown
        $dailyBreakdown = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
            ->get()
            ->groupBy(function ($payment) {
                return $payment->created_at->format('Y-m-d');
            })
            ->map(function ($payments, $date) {
                return [
                    'date' => $date,
                    'amount' => (float) $payments->sum('amount'),
                    'count' => $payments->count(),
                ];
            })
            ->sortBy('date')
            ->values();

        // Total transactions
        $totalTransactions = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
            ->count();

        return response()->json([
            'period' => [
                'from' => $dateFrom,
                'to' => $dateTo,
            ],
            'summary' => [
                'total_collected' => (float) $totalCollections,
                'total_transactions' => $totalTransactions,
                'ad_revenue' => (float) Payment::where('society_id', $society->id)
                    ->where('status', 'completed')
                    ->whereNull('bill_id')
                    ->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
                    ->sum('amount'),
                'by_method' => $byMethod,
            ],
            'daily_breakdown' => $dailyBreakdown,
        ]);
    }

    /**
     * Get arrears/bills report.
     */
    public function arrears(Request $request, Society $society)
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $dateFrom = $request->get('date_from', null);
        $dateTo = $request->get('date_to', null);

        $query = Bill::where('society_id', $society->id)
            ->with(['user:id,name,unit_id', 'user.unit:id,unit_no,building_id', 'user.unit.building:id,name']);

        // Filter by status
        if ($request->has('status') && $request->get('status') !== 'all') {
            $query->where('status', $request->get('status'));
        } elseif (!$request->has('status') && !$dateFrom) {
            // Default behavior if NO filtering params provided: show outstanding
            $query->whereIn('status', ['due', 'overdue']);
        }
        
        // Filter by Date (Created At)
        // If date range provided, filter by creation date to show "Bills Generated"
        if ($dateFrom && $dateTo) {
            $query->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59']);
        }

        // Filter by building
        if ($request->has('building_id')) {
            $query->whereHas('user.unit', function ($q) use ($request) {
                $q->where('building_id', $request->get('building_id'));
            });
        }

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('bill_no', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    })
                    ->orWhereHas('user.unit', function ($q) use ($search) {
                        $q->where('unit_no', 'like', "%{$search}%");
                    });
            });
        }

        $bills = $query->latest()->get()->map(function ($bill) {
            $daysOverdue = ($bill->status === 'due' || $bill->status === 'overdue') && $bill->due_date
                ? (now() > $bill->due_date ? now()->diffInDays($bill->due_date) : 0)
                : 0;

            return [
                'id' => $bill->id,
                'bill_no' => $bill->bill_no,
                'unit_no' => $bill->user->unit->unit_no ?? 'N/A',
                'building_name' => $bill->user->unit->building->name ?? 'N/A',
                'resident_name' => $bill->user->name,
                'amount' => (float) $bill->amount,
                'due_date' => $bill->due_date ? $bill->due_date->format('Y-m-d') : 'N/A',
                'status' => $bill->status,
                'days_overdue' => (int) $daysOverdue,
                'created_at' => $bill->created_at->format('Y-m-d'),
            ];
        });

        // Calculate totals
        // If filtering by specific status, totals reflect that.
        // If date filtered, totals reflect bills in that period.
        $totalAmount = $bills->sum('amount');
        $totalBills = $bills->count();

        return response()->json([
            'total_amount' => (float) $totalAmount,
            'total_bills' => $totalBills,
            'data' => $bills->values(),
        ]);
    }

    /**
     * Get expense breakdown report.
     */
    public function expenseBreakdown(Request $request, Society $society)
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        // Total expenses
        $totalExpenses = Expense::where('society_id', $society->id)
            ->where('expense_date', '>=', $dateFrom)
            ->where('expense_date', '<=', $dateTo . ' 23:59:59')
            ->sum('amount');

        // By category
        $byCategory = Expense::where('society_id', $society->id)
            ->where('expense_date', '>=', $dateFrom)
            ->where('expense_date', '<=', $dateTo . ' 23:59:59')
            ->get()
            ->groupBy('category')
            ->map(function ($expenses, $category) {
                return [
                    'category' => $category ?: 'Uncategorized',
                    'amount' => (float) $expenses->sum('amount'),
                    'count' => $expenses->count(),
                ];
            })
            ->sortByDesc('amount')
            ->values();

        // By vendor
        $byVendor = Expense::where('society_id', $society->id)
            ->where('expense_date', '>=', $dateFrom)
            ->where('expense_date', '<=', $dateTo . ' 23:59:59')
            ->whereNotNull('vendor_name')
            ->get()
            ->groupBy('vendor_name')
            ->map(function ($expenses, $vendor) {
                return [
                    'vendor' => $vendor,
                    'amount' => (float) $expenses->sum('amount'),
                    'count' => $expenses->count(),
                ];
            })
            ->sortByDesc('amount')
            ->values()
            ->take(10);

        // Monthly breakdown
        $monthlyBreakdown = Expense::where('society_id', $society->id)
            ->where('expense_date', '>=', $dateFrom)
            ->where('expense_date', '<=', $dateTo . ' 23:59:59')
            ->get()
            ->groupBy(function ($expense) {
                return $expense->expense_date->format('Y-m');
            })
            ->map(function ($expenses, $month) {
                return [
                    'month' => $month,
                    'amount' => (float) $expenses->sum('amount'),
                    'count' => $expenses->count(),
                ];
            })
            ->sortBy('month')
            ->values();

        return response()->json([
            'period' => [
                'from' => $dateFrom,
                'to' => $dateTo,
            ],
            'summary' => [
                'total_expenses' => (float) $totalExpenses,
                'by_category' => $byCategory,
                'by_vendor' => $byVendor,
            ],
            'monthly_breakdown' => $monthlyBreakdown,
        ]);
    }

    /**
     * Get financial summary report.
     */
    public function financialSummary(Request $request, Society $society)
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        // Collections
        $collections = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
            ->sum('amount');

        // Expenses
        $expenses = Expense::where('society_id', $society->id)
            ->where('expense_date', '>=', $dateFrom)
            ->where('expense_date', '<=', $dateTo . ' 23:59:59')
            ->where('expense_date', '<=', $dateTo . ' 23:59:59')
            ->sum('amount');

        // Ad Revenue (Subset of Collections)
        $adRevenue = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->whereNull('bill_id')
            ->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
            ->sum('amount');

        // Outstanding dues (current state)
        $outstandingDues = Bill::where('society_id', $society->id)
            ->whereIn('status', ['due', 'overdue'])
            ->sum('amount');

        // Net income
        $netIncome = $collections - $expenses;

        // Collection Rate: (Collections) / (Total Demand Created in Period)
        // Demand = Bills generated in this period
        $demandInPeriod = Bill::where('society_id', $society->id)
            ->whereBetween('created_at', [$dateFrom . ' 00:00:00', $dateTo . ' 23:59:59'])
            ->sum('amount');

        // If no bills generated, collection rate is moot (or 100% if collections exist?)
        $collectionRate = $demandInPeriod > 0 ? round(($collections / $demandInPeriod) * 100, 2) : ($collections > 0 ? 100 : 0);

        return response()->json([
            'period' => [
                'from' => $dateFrom,
                'to' => $dateTo,
            ],
            'summary' => [
                'collections' => (float) $collections,
                'ad_revenue' => (float) $adRevenue,
                'expenses' => (float) $expenses,
                'net_income' => (float) $netIncome,
                'outstanding_dues' => (float) $outstandingDues,
                'collection_rate' => $collectionRate > 100 ? 100 : $collectionRate,
            ],
        ]);
    }
}
