<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StorePaymentRequest;
use App\Models\BankAccount;
use App\Models\Bill;
use App\Models\Payment;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Inertia\Response;

class PaymentController extends Controller
{
    /**
     * Display a listing of payments.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $query = Payment::where('society_id', $society->id)
            ->with(['user:id,name,phone,unit_id', 'user.unit:id,unit_no', 'bill:id,bill_no,amount', 'bankAccount:id,account_name,bank_name']);

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('payment_no', 'like', "%{$search}%")
                    ->orWhere('transaction_id', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%");
                    });
            });
        }

        // Filter by status
        if ($request->has('status') && $request->get('status') !== 'all') {
            $query->where('status', $request->get('status'));
        }

        // Filter by payment method
        if ($request->has('payment_method') && $request->get('payment_method') !== 'all') {
            $query->where('payment_method', $request->get('payment_method'));
        }

        // Filter by resident
        if ($request->has('user_id') && $request->get('user_id') !== 'all') {
            $query->where('user_id', $request->get('user_id'));
        }

        $payments = $query->latest()->paginate(15)->withQueryString();

        // Get residents for filter
        $residentRole = Role::where('name', 'resident')->first();
        $residents = $residentRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $residentRole->id)
                ->with('unit:id,unit_no')
                ->orderBy('name')
                ->get(['id', 'name', 'phone', 'unit_id'])
            : collect();

        // Get bank accounts (include is_primary for auto-selection)
        $bankAccounts = BankAccount::where('society_id', $society->id)
            ->orderByDesc('is_primary') // Primary first
            ->orderBy('account_name')
            ->get(['id', 'account_name', 'bank_name', 'account_number', 'is_primary']);

        // Calculate stats
        $stats = [
            'total_collected' => Payment::where('society_id', $society->id)
                ->where('status', 'completed')
                ->sum('amount'),
            'pending_amount' => Payment::where('society_id', $society->id)
                ->where('status', 'pending')
                ->sum('amount'),
            'total_payments' => Payment::where('society_id', $society->id)->count(),
            'pending_reconciliation' => Payment::where('society_id', $society->id)
                ->where('status', 'pending_reconciliation')
                ->count(),
        ];

        return Inertia::render('SocietyAdmin/Payments/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'payments' => $payments,
            'residents' => $residents,
            'bankAccounts' => $bankAccounts,
            'stats' => $stats,
            'filters' => $request->only(['search', 'status', 'payment_method', 'user_id']),
        ]);
    }

    /**
     * Show the form for creating a new payment.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Get residents
        $residentRole = Role::where('name', 'resident')->first();
        $residents = $residentRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $residentRole->id)
                ->where('status', 'active')
                ->with('unit:id,unit_no')
                ->orderBy('name')
                ->get(['id', 'name', 'phone', 'unit_id'])
            : collect();

        // Get bills (due/overdue only)
        $bills = Bill::where('society_id', $society->id)
            ->whereIn('status', ['due', 'overdue'])
            ->with(['user:id,name', 'user.unit:id,unit_no'])
            ->orderBy('bill_no', 'desc')
            ->get(['id', 'bill_no', 'amount', 'user_id']);

        // Get bank accounts
        $bankAccounts = BankAccount::where('society_id', $society->id)
            ->orderBy('account_name')
            ->get(['id', 'account_name', 'bank_name', 'account_number']);

        return Inertia::render('SocietyAdmin/Payments/Create', [
            'society' => $society->only(['id', 'name', 'code']),
            'residents' => $residents,
            'bills' => $bills,
            'bankAccounts' => $bankAccounts,
        ]);
    }

    /**
     * Store a newly created payment.
     */
    public function store(StorePaymentRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;

        // Convert "none" values to null for optional fields
        if (isset($data['bill_id']) && $data['bill_id'] === 'none') {
            $data['bill_id'] = null;
        }
        if (isset($data['bank_account_id']) && $data['bank_account_id'] === 'none') {
            $data['bank_account_id'] = null;
        }

        // [NEW] Fallback to Primary Account if not specified
        if (empty($data['bank_account_id'])) {
             $primaryAccount = \App\Models\BankAccount::where('society_id', $society->id)
                ->where('is_primary', true)
                ->first();
             
             if (!$primaryAccount) {
                 $primaryAccount = \App\Models\BankAccount::where('society_id', $society->id)->first();
             }

             if ($primaryAccount) {
                 $data['bank_account_id'] = $primaryAccount->id;
             }
        }

        DB::transaction(function () use ($data, $society) {
            // Handle multiple bills selection (Quick Collect)
            if (!empty($data['bill_ids']) && is_array($data['bill_ids'])) {
                $bills = Bill::whereIn('id', $data['bill_ids'])->get();
                $basePaymentNo = $data['payment_no'] ?? null;
                
                foreach ($bills as $index => $bill) {
                    $paymentData = $data;
                    $paymentData['bill_id'] = $bill->id;
                    $paymentData['amount'] = $bill->amount; // Allocate exact bill amount
                    
                    // Generate Payment No
                    if (empty($basePaymentNo)) {
                        $paymentNoPrefix = 'PAY-'.strtoupper(substr($society->code ?? 'SOC', 0, 3)).'-'.date('Y').'-';
                        // Use time + index + fractional seconds to ensure uniqueness
                        $uniqueSuffix = str_pad((string)(time() + $index), 8, '0', STR_PAD_LEFT) . str_pad((string)$index, 2, '0', STR_PAD_LEFT);
                        $paymentData['payment_no'] = $paymentNoPrefix.$uniqueSuffix;
                    } else {
                        $paymentData['payment_no'] = $basePaymentNo . '-' . ($index + 1);
                    }

                    $paymentData['collected_by'] = Auth::id();
                    $paymentData['collector_role'] = 'admin';
                    
                    $payment = Payment::create($paymentData);

                    // Record Transaction ONLY for completed payments
                    if ($payment->bank_account_id && $data['status'] === 'completed') {
                        app(\App\Services\TransactionService::class)->recordIncome(
                            $payment->amount,
                            $payment->bank_account_id,
                            'bill_payment',
                            $payment,
                            $payment->payment_method,
                            'Payment recorded by Admin',
                            Auth::id()
                        );
                    }

                    // Immediately update bill status if completed
                    if ($data['status'] === 'completed') {
                        $bill->update([
                            'status' => 'paid',
                            'paid_at' => now(),
                        ]);
                    }
                }
            } else {
                // Handle Single Payment (Manual Entry or Single Bill)
                if (empty($data['payment_no'])) {
                    $paymentNoPrefix = 'PAY-'.strtoupper(substr($society->code ?? 'SOC', 0, 3)).'-'.date('Y').'-';
                    $paymentNo = $paymentNoPrefix.str_pad(time(), 8, '0', STR_PAD_LEFT);
                    $data['payment_no'] = $paymentNo;
                }

                $data['collected_by'] = Auth::id();
                $data['collector_role'] = 'admin';

                $payment = Payment::create($data);

                // Record Transaction ONLY for completed payments
                if ($payment->bank_account_id && $data['status'] === 'completed') {
                    app(\App\Services\TransactionService::class)->recordIncome(
                        $payment->amount,
                        $payment->bank_account_id,
                        'bill_payment',
                        $payment,
                        $payment->payment_method,
                        'Payment recorded by Admin',
                        Auth::id()
                    );
                }

                // Update bill status if bill is associated
                if (! empty($data['bill_id'])) {
                    $bill = Bill::find($data['bill_id']);

                    if ($bill && $bill->society_id === $society->id) {
                        // Check if payment amount covers the bill
                        $totalPayments = Payment::where('bill_id', $bill->id)
                            ->where('status', 'completed')
                            ->sum('amount');

                        if ($data['status'] === 'completed' && ($totalPayments + $data['amount']) >= $bill->amount) {
                            $bill->update([
                                'status' => 'paid',
                                'paid_at' => now(),
                            ]);
                        }
                    }
                }
            }
        });

        return redirect()->route('society.payments.index', $society)
            ->with('success', 'Payment recorded successfully.');
    }

    /**
     * Display the specified payment.
     */
    public function show(Society $society, Payment $payment): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $payment->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $payment->load([
            'user:id,name,phone,email,unit_id',
            'user.unit:id,unit_no,building_id',
            'user.unit.building:id,name',
            'bill:id,bill_no,amount,status',
            'bankAccount:id,account_name,bank_name,account_number',
            'reconciledBy:id,name',
        ]);

        return Inertia::render('SocietyAdmin/Payments/Show', [
            'society' => $society->only(['id', 'name', 'code']),
            'payment' => $payment,
        ]);
    }

    /**
     * Update the specified payment.
     */
    public function update(Request $request, Society $society, Payment $payment): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $payment->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $validated = $request->validate([
            'status' => 'required|in:pending,completed,failed,pending_reconciliation',
        ]);

        $previousStatus = $payment->status;
        
        $payment->update([
            'status' => $validated['status'],
        ]);
        
        // Record transaction ONLY when status changes TO completed (and wasn't already completed)
        if ($validated['status'] === 'completed' && $previousStatus !== 'completed' && $payment->bank_account_id) {
            app(\App\Services\TransactionService::class)->recordIncome(
                $payment->amount,
                $payment->bank_account_id,
                'bill_payment',
                $payment,
                $payment->payment_method,
                'Payment marked as completed',
                Auth::id()
            );
        }
        
        // If payment is completed and linked to a bill, check if bill should be marked as paid
        if ($validated['status'] === 'completed' && $payment->bill_id) {
            $bill = Bill::find($payment->bill_id);
            if ($bill) {
                $totalPayments = Payment::where('bill_id', $bill->id)
                    ->where('status', 'completed')
                    ->sum('amount');
                
                if ($totalPayments >= $bill->amount) {
                    $bill->update([
                        'status' => 'paid',
                        'paid_at' => now(),
                    ]);
                }
            }
        }

        return redirect()->back()->with('success', 'Payment status updated successfully.');
    }

    /**
     * Reconcile the specified payment.
     */
    public function reconcile(Society $society, Payment $payment): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $payment->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        if ($payment->status !== 'completed') {
            return redirect()->back()
                ->with('error', 'Only completed payments can be reconciled.');
        }

        $payment->update([
            'reconciled_by' => $user->id,
            'reconciled_at' => now(),
            'status' => 'completed',
        ]);

        return redirect()->route('society.payments.show', [$society, $payment])
            ->with('success', 'Payment reconciled successfully.');
    }

    /**
     * Get due bills for a specific resident.
     */
    public function getDueBills(Request $request, Society $society, User $resident)
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $resident->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $dueBills = Bill::where('society_id', $society->id)
            ->where('user_id', $resident->id)
            ->whereIn('status', ['pending', 'approved', 'due', 'overdue'])
            ->orderBy('due_date', 'asc')
            ->get(['id', 'bill_no', 'amount', 'status', 'due_date', 'maintenance', 'water', 'electricity', 'parking', 'other_charges', 'gst', 'discount']);

        return response()->json([
            'dueBills' => $dueBills,
        ]);
    }

    /**
     * Generate Razorpay QR Code for payment.
     */
    public function generateQRCode(Request $request, Society $society)
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $amount = $request->input('amount');
        $userId = $request->input('user_id');

        if (!$amount || $amount <= 0) {
            return response()->json(['error' => 'Invalid amount'], 400);
        }

        // Load society settings
        $society->load('settings');
        
        // Get Razorpay credentials from society settings
        $societySettings = $society->settings;
        
        if (!$societySettings || !$societySettings->razorpay_enabled) {
            return response()->json(['error' => 'Razorpay is not enabled for this society'], 400);
        }

        $razorpayKey = $societySettings->razorpay_key_id;
        $razorpaySecret = $societySettings->razorpay_key_secret;

        if (!$razorpayKey || !$razorpaySecret) {
            return response()->json(['error' => 'Razorpay credentials not configured'], 500);
        }

        try {
            $api = new \Razorpay\Api\Api($razorpayKey, $razorpaySecret);

            // Create Razorpay Order
            $orderData = [
                'receipt' => 'rcpt_'.time(),
                'amount' => $amount * 100, // Amount in paise
                'currency' => 'INR',
                'notes' => [
                    'society_id' => $society->id,
                    'user_id' => $userId,
                ],
            ];

            $order = $api->order->create($orderData);

            // Generate QR Code
            $qrData = [
                'type' => 'upi_qr',
                'name' => $society->name.' Payment',
                'usage' => 'single_use',
                'fixed_amount' => true,
                'payment_amount' => $amount * 100,
                'description' => 'Payment for '.$society->name,
                'customer_id' => 'cust_'.time(),
                'close_by' => time() + 3600, // QR valid for 1 hour
                'notes' => [
                    'society_id' => $society->id,
                    'user_id' => $userId,
                    'order_id' => $order->id,
                ],
            ];

            $qrCode = $api->qrCode->create($qrData);

            return response()->json([
                'order_id' => $order->id,
                'qr_code_url' => $qrCode->image_url,
                'qr_code_id' => $qrCode->id,
            ]);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }
    }

    /**
     * Send Payment Request (Payment Link) to Resident's Mobile.
     */
    public function sendPaymentRequest(Request $request, Society $society)
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $request->validate([
            'user_id' => 'required|exists:users,id',
            'amount' => 'required|numeric|min:1',
            'bill_id' => 'nullable|exists:bills,id',
            'description' => 'nullable|string|max:255',
        ]);

        $residentId = $request->input('user_id');
        $amount = $request->input('amount');
        $billId = $request->input('bill_id');
        $description = $request->input('description') ?? 'Payment Request from ' . $society->name;

        // Load resident
        // Validate resident belongs to the society
        $resident = User::find($residentId);
        if (!$resident || $resident->society_id !== $society->id) {
            \Log::error('sendPaymentRequest: Invalid resident', [
                'society_id' => $society->id,
                'requested_user_id' => $residentId,
            ]);
            return response()->json(['error' => 'Invalid resident'], 400);
        }

        // Load society settings for payment gateway configuration
        $society->load('settings');
        $societySettings = $society->settings;

        // Validate Razorpay is enabled and configured
        if (!$societySettings || !$societySettings->razorpay_enabled) {
            \Log::error('sendPaymentRequest: Razorpay not enabled', [
                'society_id' => $society->id,
            ]);
            return response()->json(['error' => 'Razorpay is not enabled for this society. Please configure it in Society Settings.'], 400);
        }

        $razorpayKey = $societySettings->razorpay_key_id;
        $razorpaySecret = $societySettings->razorpay_key_secret;

        if (!$razorpayKey || !$razorpaySecret) {
            \Log::error('sendPaymentRequest: Razorpay credentials missing', [
                'society_id' => $society->id,
            ]);
            return response()->json(['error' => 'Razorpay credentials not configured in Society Settings'], 500);
        }

        // Sanitize and format phone number for Razorpay (must be in E.164 format)
        $rawPhone = $resident->phone ?? '';
        $sanitizedPhone = preg_replace('/\D/', '', $rawPhone);

        if (strlen($sanitizedPhone) === 12 && str_starts_with($sanitizedPhone, '91')) {
            $sanitizedPhone = '+' . $sanitizedPhone;
        } elseif (strlen($sanitizedPhone) === 10) {
            $sanitizedPhone = '+91' . $sanitizedPhone;
        }

        try {
            $api = new \Razorpay\Api\Api($razorpayKey, $razorpaySecret);
            $referenceId = 'PAYREQ_ADMIN_' . time() . '_' . ($billId ?? 'GEN');

            $linkData = [
                'amount' => $amount * 100, // in paise
                'currency' => 'INR',
                'accept_partial' => false,
                'first_min_partial_amount' => 0,
                'description' => $description,
                'customer' => [
                    'name' => $resident->name,
                    'contact' => $sanitizedPhone,
                    'email' => $resident->email ?? 'noreply@societify.com',
                ],
                'notify' => [
                    'sms' => true,
                    'email' => !empty($resident->email),
                ],
                'reminder_enable' => true,
                'notes' => [
                    'society_id' => (string) $society->id,
                    'user_id' => (string) $resident->id,
                    'bill_id' => (string) ($billId ?? ''),
                    'admin_id' => (string) $user->id,
                    'reference_id' => $referenceId,
                ],
            ];

            // Create Razorpay payment link
            $link = $api->paymentLink->create($linkData);

            // Create Pending Payment Record (with primary bank account for later transaction recording)
            $primaryAccount = BankAccount::where('society_id', $society->id)
                ->where('is_primary', true)
                ->first() ?? BankAccount::where('society_id', $society->id)->first();

            $payment = Payment::create([
                'society_id' => $society->id,
                'bill_id' => $billId,
                'user_id' => $resident->id,
                'payment_no' => $referenceId,
                'amount' => $amount,
                'payment_method' => 'upi_request',
                'status' => 'pending',
                'payment_request_id' => $link->id,
                'bank_account_id' => $primaryAccount?->id,
                'notes' => 'Payment Request sent by Admin via Payment Link',
                'collected_by' => $user->id,
                'collector_role' => 'admin',
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Payment request sent to ' . $resident->name,
                'payment_link' => $link->short_url,
                'link_id' => $link->id,
                'payment_id' => $payment->id,
            ]);

        } catch (\Exception $e) {
            // Log error for debugging payment gateway issues
            \Log::error('sendPaymentRequest: Razorpay error', [
                'society_id' => $society->id,
                'resident_id' => $resident->id,
                'error_message' => $e->getMessage(),
                'error_code' => $e->getCode(),
            ]);
            return response()->json(['error' => 'Razorpay Error: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Check Payment Link Status from Razorpay
     */
    public function checkPaymentStatus(Society $society, $linkId)
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Load society settings
        $society->load('settings');
        $societySettings = $society->settings;

        if (!$societySettings || !$societySettings->razorpay_enabled) {
            return response()->json(['error' => 'Razorpay is not enabled'], 400);
        }

        $razorpayKey = $societySettings->razorpay_key_id;
        $razorpaySecret = $societySettings->razorpay_key_secret;

        if (!$razorpayKey || !$razorpaySecret) {
            return response()->json(['error' => 'Razorpay credentials not configured'], 500);
        }

        try {
            $api = new \Razorpay\Api\Api($razorpayKey, $razorpaySecret);
            
            // Fetch payment link status from Razorpay
            $link = $api->paymentLink->fetch($linkId);

            // Update local payment record if status changed
            if ($link->status === 'paid') {
                $payment = Payment::where('payment_request_id', $linkId)->first();
                if ($payment && $payment->status !== 'completed') {
                    $payment->update([
                        'status' => 'completed',
                        'transaction_id' => $link->payments[0]['payment_id'] ?? null,
                    ]);

                    // Update associated bill if exists
                    if ($payment->bill_id) {
                        $bill = $payment->bill;
                        if ($bill) {
                            $totalPaid = Payment::where('bill_id', $bill->id)
                                ->where('status', 'completed')
                                ->sum('amount');
                            
                            if ($totalPaid >= $bill->amount) {
                                $bill->update(['status' => 'paid']);
                            } else {
                                $bill->update(['status' => 'partial']);
                            }
                        }
                    }
                }
            }

            return response()->json([
                'status' => $link->status,
                'amount' => $link->amount / 100,
                'short_url' => $link->short_url,
            ]);

        } catch (\Exception $e) {
            // Log error for debugging payment status check issues
            \Log::error('checkPaymentStatus: Failed to check payment link status', [
                'link_id' => $linkId,
                'error' => $e->getMessage(),
            ]);
            return response()->json(['error' => 'Unable to check status: ' . $e->getMessage()], 500);
        }
    }
}
