<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreMaintenanceTicketRequest;
use App\Models\MaintenanceTicket;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class MaintenanceController extends Controller
{
    /**
     * Display a listing of maintenance tickets (Kanban view).
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $query = MaintenanceTicket::where('society_id', $society->id)
            ->with(['user:id,name,phone,unit_id', 'user.unit:id,unit_no', 'vendor:id,name,phone']);

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('ticket_no', 'like', "%{$search}%")
                    ->orWhere('title', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%");
                    });
            });
        }

        // Filter by priority
        if ($request->has('priority') && $request->get('priority') !== 'all') {
            $query->where('priority', $request->get('priority'));
        }

        // Filter by vendor
        if ($request->has('vendor_id') && $request->get('vendor_id') !== 'all') {
            $query->where('vendor_id', $request->get('vendor_id'));
        }



        // Get vendors for filter
        $vendorRole = Role::where('name', 'vendor')->first();
        $vendors = $vendorRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $vendorRole->id)
                ->where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'phone'])
            : collect();

        // Get residents for ticket creation
        $residentRole = Role::where('name', 'resident')->first();
        $residents = $residentRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $residentRole->id)
                ->where('status', 'active')
                ->with('unit:id,unit_no')
                ->orderBy('name')
                ->get(['id', 'name', 'phone', 'unit_id'])
            : collect();

        // Calculate Stats
        $totalTickets = MaintenanceTicket::where('society_id', $society->id)->count();
        $openTickets = MaintenanceTicket::where('society_id', $society->id)->where('status', 'open')->count();
        $inProgressTickets = MaintenanceTicket::where('society_id', $society->id)->where('status', 'in_progress')->count();
        $resolvedTickets = MaintenanceTicket::where('society_id', $society->id)->where('status', 'resolved')->count();

        // Get tickets for Table View (paginated)
        $tickets = $query->latest()->paginate(15)->withQueryString();

        return Inertia::render('SocietyAdmin/Maintenance/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'tickets' => $tickets,
            'vendors' => $vendors,
            'residents' => $residents,
            'filters' => $request->only(['search', 'priority', 'vendor_id']),
            'stats' => [
                'total' => $totalTickets,
                'open' => $openTickets,
                'in_progress' => $inProgressTickets,
                'resolved' => $resolvedTickets,
            ],
        ]);
    }

    /**
     * Show the form for creating a new maintenance ticket.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Get residents
        $residentRole = Role::where('name', 'resident')->first();
        $residents = $residentRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $residentRole->id)
                ->where('status', 'active')
                ->with('unit:id,unit_no')
                ->orderBy('name')
                ->get(['id', 'name', 'phone', 'unit_id'])
            : collect();

        // Get vendors
        $vendorRole = Role::where('name', 'vendor')->first();
        $vendors = $vendorRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $vendorRole->id)
                ->where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'phone'])
            : collect();

        return Inertia::render('SocietyAdmin/Maintenance/Create', [
            'society' => $society->only(['id', 'name', 'code']),
            'residents' => $residents,
            'vendors' => $vendors,
        ]);
    }

    /**
     * Store a newly created maintenance ticket.
     */
    public function store(StoreMaintenanceTicketRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;

        // Convert "none" values to null for optional fields
        if (isset($data['vendor_id']) && $data['vendor_id'] === 'none') {
            $data['vendor_id'] = null;
        }

        // Generate ticket number if not provided
        if (empty($data['ticket_no'])) {
            $ticketNoPrefix = 'TKT-'.strtoupper(substr($society->code ?? 'SOC', 0, 3)).'-'.date('Y').'-';
            $ticketNo = $ticketNoPrefix.str_pad(time(), 8, '0', STR_PAD_LEFT);
            $data['ticket_no'] = $ticketNo;
        }

        // Set assigned_at if vendor is assigned
        if (! empty($data['vendor_id'])) {
            $data['assigned_at'] = now();
            $data['status'] = 'assigned';
        }

        MaintenanceTicket::create($data);

        return redirect()->route('society.maintenance.index', $society)
            ->with('success', 'Maintenance ticket created successfully.');
    }

    /**
     * Display the specified maintenance ticket.
     */
    public function show(Society $society, MaintenanceTicket $maintenanceTicket): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $maintenanceTicket->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $maintenanceTicket->load([
            'user:id,name,phone,email,unit_id',
            'user.unit:id,unit_no,building_id',
            'user.unit.building:id,name',
            'vendor:id,name,phone',
            'society:id,name',
            'comments.user:id,name,role_id',
            'comments.user.role:id,name',
            'timelines.user:id,name',
        ]);

        // Merge Timelines and Comments uniquely
        $history = collect();

        foreach ($maintenanceTicket->timelines as $timeline) {
            $history->push([
                'type' => 'timeline',
                'id' => $timeline->id,
                'action' => $timeline->action,
                'description' => $timeline->description,
                'timestamp' => $timeline->created_at,
                'user' => $timeline->user ? $timeline->user->name : 'System',
            ]);
        }

        foreach ($maintenanceTicket->comments as $comment) {
            $history->push([
                'type' => 'comment',
                'id' => $comment->id,
                'message' => $comment->message,
                'images' => $comment->images,
                'timestamp' => $comment->created_at,
                'user' => $comment->user->name,
                'user_role' => $comment->user->role->name ?? 'User',
                'is_me' => $comment->user_id === $user->id,
            ]);
        }

        $sortedHistory = $history->sortByDesc('timestamp')->values();

        // Get vendors for assignment (reused logic)
        $vendorRole = Role::where('name', 'vendor')->first();
        $vendors = $vendorRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $vendorRole->id)
                ->where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'phone'])
            : collect();

        return Inertia::render('SocietyAdmin/Maintenance/Show', [
            'society' => $society->only(['id', 'name', 'code']),
            'ticket' => $maintenanceTicket,
            'history' => $sortedHistory,
            'vendors' => $vendors,
        ]);
    }

    /**
     * Update the status of a maintenance ticket.
     */
    public function updateStatus(Request $request, Society $society, MaintenanceTicket $maintenanceTicket): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $maintenanceTicket->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $request->validate([
            'status' => ['required', 'in:open,assigned,in_progress,resolved,closed'],
        ]);

        $data = ['status' => $request->get('status')];

        // Set timestamps based on status
        if ($request->get('status') === 'assigned' && ! $maintenanceTicket->assigned_at) {
            $data['assigned_at'] = now();
        }

        if ($request->get('status') === 'resolved' && ! $maintenanceTicket->resolved_at) {
            $data['resolved_at'] = now();
        }

        $maintenanceTicket->update($data);

        return redirect()->back()
            ->with('success', 'Ticket status updated successfully.');
    }

    /**
     * Assign a vendor to a maintenance ticket.
     */
    public function assignVendor(Request $request, Society $society, MaintenanceTicket $maintenanceTicket): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $maintenanceTicket->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $vendorId = $request->get('vendor_id');

        // Handle "none" value
        if ($vendorId === 'none' || empty($vendorId)) {
            return redirect()->back()
                ->withErrors(['vendor_id' => 'Please select a valid vendor.']);
        }

        $request->validate([
            'vendor_id' => ['required', 'exists:users,id'],
        ]);

        $maintenanceTicket->update([
            'vendor_id' => $vendorId,
            'status' => 'assigned',
            'assigned_at' => now(),
        ]);

        return redirect()->back()
            ->with('success', 'Vendor assigned successfully.');
    }

    /**
     * Resolve a maintenance ticket.
     */
    public function resolve(Request $request, Society $society, MaintenanceTicket $maintenanceTicket): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $maintenanceTicket->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $request->validate([
            'resolution_notes' => ['nullable', 'string'],
        ]);

        $maintenanceTicket->update([
            'status' => 'resolved',
            'resolved_at' => now(),
            'resolution_notes' => $request->get('resolution_notes'),
        ]);

        return redirect()->back()
            ->with('success', 'Ticket resolved successfully.');
    }
}
