<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Models\Facility;
use App\Models\FacilityBooking;
use App\Models\FacilitySlot;
use App\Models\Society;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;

class FacilityController extends Controller
{
    /**
     * Display a listing of the facilities.
     */
    public function index(Society $society)
    {
        $facilities = $society->facilities()
            ->withCount('bookings')
            ->with('slots')
            ->latest()
            ->paginate(15);

        return Inertia::render('SocietyAdmin/Facilities/Index', [
            'society' => $society,
            'facilities' => $facilities,
        ]);
    }

    /**
     * Show the form for creating a new facility.
     */
    public function create(Society $society)
    {
        return Inertia::render('SocietyAdmin/Facilities/Create', [
            'society' => $society,
        ]);
    }

    /**
     * Store a newly created facility in storage.
     */
    public function store(Request $request, Society $society)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'images' => 'nullable|array',
            'images.*' => 'url',
            'capacity' => 'nullable|integer|min:1',
            'amenities' => 'nullable|string',
            'rules' => 'nullable|string',
            'cancellation_policy' => 'nullable|string',
            'requires_approval' => 'boolean',
        ]);

        $facility = $society->facilities()->create([
            ...$validated,
            'is_active' => true,
        ]);

        return redirect()
            ->route('society.facilities.show', ['society' => $society, 'facility' => $facility])
            ->with('success', 'Facility created successfully');
    }

    /**
     * Display the specified facility.
     */
    public function show(Society $society, Facility $facility)
    {
        // Load all bookings for calendar view (future and recent past)
        $facility->load(['slots', 'bookings' => function ($query) {
            $query->with(['user', 'bookedBy', 'slot'])
                ->where('booking_date', '>=', now()->subMonths(1)->startOfMonth()) // Last month onwards
                ->orderBy('booking_date', 'asc');
        }]);

        // Get booking statistics
        $stats = [
            'total_bookings' => $facility->bookings()->count(),
            'pending_approval' => $facility->bookings()->where('booking_status', 'pending')->count(),
            'confirmed' => $facility->bookings()->where('booking_status', 'confirmed')->count(),
            'revenue_collected' => $facility->bookings()
                ->where('payment_status', 'paid')
                ->sum('amount'),
            'revenue_pending' => $facility->bookings()
                ->where('payment_status', 'pending')
                ->sum('amount'),
        ];

        return Inertia::render('SocietyAdmin/Facilities/Show', [
            'society' => $society,
            'facility' => $facility,
            'stats' => $stats,
        ]);
    }

    /**
     * Show the form for editing the specified facility.
     */
    public function edit(Society $society, Facility $facility)
    {
        return Inertia::render('SocietyAdmin/Facilities/Edit', [
            'society' => $society,
            'facility' => $facility->load('slots'),
        ]);
    }

    /**
     * Update the specified facility in storage.
     */
    public function update(Request $request, Society $society, Facility $facility)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'images' => 'nullable|array',
            'images.*' => 'url',
            'capacity' => 'nullable|integer|min:1',
            'amenities' => 'nullable|string',
            'rules' => 'nullable|string',
            'cancellation_policy' => 'nullable|string',
            'requires_approval' => 'boolean',
            'is_active' => 'boolean',
        ]);

        $facility->update($validated);

        return redirect()
            ->route('society.facilities.show', ['society' => $society, 'facility' => $facility])
            ->with('success', 'Facility updated successfully');
    }

    /**
     * Remove the specified facility from storage.
     */
    public function destroy(Society $society, Facility $facility)
    {
        // Check if there are active bookings
        $hasActiveBookings = $facility->bookings()
            ->whereIn('booking_status', ['pending', 'confirmed'])
            ->exists();

        if ($hasActiveBookings) {
            return back()->with('error', 'Cannot delete facility with active bookings');
        }

        $facility->delete();

        return redirect()
            ->route('society.facilities.index', $society)
            ->with('success', 'Facility deleted successfully');
    }

    /**
     * Manage slots for a facility.
     */
    public function manageSlots(Request $request, Society $society, Facility $facility)
    {
        $validated = $request->validate([
            'slots' => 'required|array',
            'slots.*.id' => 'nullable|exists:facility_slots,id',
            'slots.*.name' => 'required|string|max:255',
            'slots.*.start_time' => 'required|date_format:H:i',
            'slots.*.end_time' => 'required|date_format:H:i|after:slots.*.start_time',
            'slots.*.price' => 'required|numeric|min:0',
            'slots.*.is_active' => 'boolean',
        ]);

        foreach ($validated['slots'] as $slotData) {
            if (isset($slotData['id'])) {
                // Update existing slot
                $slot = FacilitySlot::where('facility_id', $facility->id)
                    ->find($slotData['id']);
                
                if ($slot) {
                    $slot->update([
                        'name' => $slotData['name'],
                        'start_time' => $slotData['start_time'],
                        'end_time' => $slotData['end_time'],
                        'price' => $slotData['price'],
                        'is_active' => $slotData['is_active'] ?? true,
                    ]);
                }
            } else {
                // Create new slot
                $facility->slots()->create([
                    'name' => $slotData['name'],
                    'start_time' => $slotData['start_time'],
                    'end_time' => $slotData['end_time'],
                    'price' => $slotData['price'],
                    'is_active' => $slotData['is_active'] ?? true,
                ]);
            }
        }

        return back()->with('success', 'Slots updated successfully');
    }

    /**
     * Delete a specific slot.
     */
    public function deleteSlot(Society $society, Facility $facility, FacilitySlot $slot)
    {
        // Check if there are active bookings for this slot
        $hasActiveBookings = $slot->bookings()
            ->whereIn('booking_status', ['pending', 'confirmed'])
            ->where('booking_date', '>=', now()->toDateString())
            ->exists();

        if ($hasActiveBookings) {
            return back()->with('error', 'Cannot delete slot with active bookings');
        }

        $slot->delete();

        return back()->with('success', 'Slot deleted successfully');
    }

    /**
     * Display all bookings for the society.
     */
    public function allBookings(Society $society, Request $request)
    {
        $query = FacilityBooking::where('society_id', $society->id)
            ->with(['facility', 'slot', 'user', 'bookedBy']);

        // Filter by facility
        if ($request->has('facility_id')) {
            $query->where('facility_id', $request->facility_id);
        }

        // Filter by status
        if ($request->has('booking_status')) {
            $query->where('booking_status', $request->booking_status);
        }

        // Filter by payment status
        if ($request->has('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        // Filter by date range
        if ($request->has('from_date')) {
            $query->where('booking_date', '>=', $request->from_date);
        }
        if ($request->has('to_date')) {
            $query->where('booking_date', '<=', $request->to_date);
        }

        $bookings = $query->latest()->paginate(20);
        $facilities = $society->facilities()->get(['id', 'name']);

        return Inertia::render('SocietyAdmin/Facilities/Bookings', [
            'society' => $society,
            'bookings' => $bookings,
            'facilities' => $facilities,
            'filters' => $request->only(['facility_id', 'booking_status', 'payment_status', 'from_date', 'to_date']),
        ]);
    }

    /**
     * Show a specific booking.
     */
    public function showBooking(Society $society, FacilityBooking $booking)
    {
        $booking->load(['facility', 'slot', 'user', 'bookedBy']);

        return Inertia::render('SocietyAdmin/Facilities/BookingDetail', [
            'booking' => $booking,
        ]);
    }

    /**
     * Update booking status.
     */
    public function updateBookingStatus(Request $request, Society $society, FacilityBooking $booking)
    {
        $validated = $request->validate([
            'booking_status' => 'required|in:pending,confirmed,cancelled,rejected,completed',
            'admin_notes' => 'nullable|string',
        ]);

        $booking->update($validated);

        return back()->with('success', 'Booking status updated successfully');
    }
}
