<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreExpenseRequest;
use App\Http\Requests\SocietyAdmin\UpdateExpenseRequest;
use App\Models\Expense;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class ExpenseController extends Controller
{
    /**
     * Display a listing of expenses.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $query = Expense::where('society_id', $society->id)
            ->with(['createdBy:id,name', 'vendor:id,name,phone']);

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('expense_no', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%")
                    ->orWhere('vendor_name', 'like', "%{$search}%")
                    ->orWhere('category', 'like', "%{$search}%");
            });
        }

        // Filter by category
        if ($request->has('category') && $request->get('category') !== 'all') {
            $query->where('category', $request->get('category'));
        }

        // Filter by date range
        if ($request->has('date_from')) {
            $query->where('expense_date', '>=', $request->get('date_from'));
        }
        if ($request->has('date_to')) {
            $query->where('expense_date', '<=', $request->get('date_to'));
        }

        // Filter by vendor
        if ($request->has('vendor_id') && $request->get('vendor_id') !== 'all') {
            $query->where('vendor_id', $request->get('vendor_id'));
        }

        $expenses = $query->latest('expense_date')->paginate(15)->withQueryString();

        // Get unique categories for filter
        $categories = Expense::where('society_id', $society->id)
            ->whereNotNull('category')
            ->distinct()
            ->pluck('category')
            ->sort()
            ->values();

        // Get vendors for filter
        $vendorRole = Role::where('name', 'vendor')->first();
        $vendors = $vendorRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $vendorRole->id)
                ->orderBy('name')
                ->get(['id', 'name', 'phone'])
            : collect();

        // Calculate stats
        $stats = [
            'total_expenses' => Expense::where('society_id', $society->id)->sum('amount'),
            'monthly_expenses' => Expense::where('society_id', $society->id)
                ->whereYear('expense_date', now()->year)
                ->whereMonth('expense_date', now()->month)
                ->sum('amount'),
            'total_count' => Expense::where('society_id', $society->id)->count(),
            'monthly_count' => Expense::where('society_id', $society->id)
                ->whereYear('expense_date', now()->year)
                ->whereMonth('expense_date', now()->month)
                ->count(),
        ];

        // Get bank accounts for expense payment
        $bankAccounts = \App\Models\BankAccount::where('society_id', $society->id)
            ->orderBy('account_name')
            ->get(['id', 'account_name', 'bank_name', 'account_number']);

        return Inertia::render('SocietyAdmin/Expenses/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'expenses' => $expenses,
            'categories' => $categories,
            'vendors' => $vendors,
            'bankAccounts' => $bankAccounts,
            'filters' => $request->only(['search', 'category', 'date_from', 'date_to', 'vendor_id']),
            'stats' => $stats,
        ]);
    }

    /**
     * Show the form for creating a new expense.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Get unique categories
        $categories = Expense::where('society_id', $society->id)
            ->whereNotNull('category')
            ->distinct()
            ->pluck('category')
            ->sort()
            ->values();

        // Get vendors
        $vendorRole = Role::where('name', 'vendor')->first();
        $vendors = $vendorRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $vendorRole->id)
                ->where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'phone'])
            : collect();

        return Inertia::render('SocietyAdmin/Expenses/Create', [
            'society' => $society->only(['id', 'name', 'code']),
            'categories' => $categories,
            'vendors' => $vendors,
        ]);
    }

    /**
     * Store a newly created expense.
     */
    public function store(StoreExpenseRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;
        $data['created_by'] = $user->id;

        // Convert "none" values to null for optional fields
        if (isset($data['vendor_id']) && $data['vendor_id'] === 'none') {
            $data['vendor_id'] = null;
        }

        // Generate expense number if not provided
        if (empty($data['expense_no'])) {
            $expenseNoPrefix = 'EXP-'.strtoupper(substr($society->code ?? 'SOC', 0, 3)).'-'.date('Y').'-';
            $expenseNo = $expenseNoPrefix.str_pad(time(), 8, '0', STR_PAD_LEFT);
            $data['expense_no'] = $expenseNo;
        }

        $expense = Expense::create($data);

        // Record Transaction
        if ($expense->bank_account_id) {
            app(\App\Services\TransactionService::class)->recordExpense(
                $expense->amount,
                $expense->bank_account_id,
                'expense',
                $expense,
                'expense', // Payment method 'expense' or based on input? Using generic 'expense' or 'bill_payment' equivalent.
                $expense->description,
                Auth::id()
            );
        }

        return redirect()->route('society.expenses.index', $society)
            ->with('success', 'Expense recorded successfully.');
    }

    /**
     * Display the specified expense.
     */
    public function show(Society $society, Expense $expense): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $expense->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: View only, no show page
        if ($user->hasRole('sub_admin') && ! $user->hasPermission('expenses.view')) {
            abort(403, 'You do not have permission to view expense details.');
        }

        $expense->load(['createdBy:id,name', 'vendor:id,name,phone', 'society:id,name']);

        return Inertia::render('SocietyAdmin/Expenses/Show', [
            'society' => $society->only(['id', 'name', 'code']),
            'expense' => $expense,
        ]);
    }

    /**
     * Show the form for editing the specified expense.
     */
    public function edit(Society $society, Expense $expense): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $expense->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No edit permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view and create expenses.');
        }

        $expense->load(['createdBy:id,name', 'vendor:id,name']);

        // Get unique categories
        $categories = Expense::where('society_id', $society->id)
            ->whereNotNull('category')
            ->distinct()
            ->pluck('category')
            ->sort()
            ->values();

        // Get vendors
        $vendorRole = Role::where('name', 'vendor')->first();
        $vendors = $vendorRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $vendorRole->id)
                ->where('status', 'active')
                ->orderBy('name')
                ->get(['id', 'name', 'phone'])
            : collect();

        return Inertia::render('SocietyAdmin/Expenses/Edit', [
            'society' => $society->only(['id', 'name', 'code']),
            'expense' => $expense,
            'categories' => $categories,
            'vendors' => $vendors,
        ]);
    }

    /**
     * Update the specified expense.
     */
    public function update(UpdateExpenseRequest $request, Society $society, Expense $expense): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $expense->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No update permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view and create expenses.');
        }

        $expense->update($request->validated());

        return redirect()->route('society.expenses.index', $society)
            ->with('success', 'Expense updated successfully.');
    }

    /**
     * Remove the specified expense.
     */
    public function destroy(Society $society, Expense $expense): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $expense->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No delete permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view and create expenses.');
        }

        $expense->delete();

        return redirect()->route('society.expenses.index', $society)
            ->with('success', 'Expense deleted successfully.');
    }
}
