<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\Society;
use App\Models\Unit;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class DirectoryController extends Controller
{
    /**
     * Display the directory of residents who have opted in.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        // Ensure user belongs to this society
        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $residentRole = Role::where('name', 'resident')->first();

        if (! $residentRole) {
            abort(500, 'Resident role not found. Please seed roles first.');
        }

        // Only show residents who have opted into directory
        $query = User::where('society_id', $society->id)
            ->where('role_id', $residentRole->id)
            ->where('directory_visible', true)
            ->where('status', 'active')
            ->with(['unit:id,unit_no,building_id', 'unit.building:id,name']);

        // Search by name, phone, email, or unit number
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('phone', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhereHas('unit', function ($q) use ($search) {
                        $q->where('unit_no', 'like', "%{$search}%");
                    });
            });
        }

        // Filter by building
        if ($request->has('building_id') && $request->get('building_id') !== 'all') {
            $query->whereHas('unit', function ($q) use ($request) {
                $q->where('building_id', $request->get('building_id'));
            });
        }

        // Filter by unit
        if ($request->has('unit_id') && $request->get('unit_id') !== 'all') {
            $query->where('unit_id', $request->get('unit_id'));
        }

        // Sort alphabetically by name
        $residents = $query->orderBy('name')->paginate(20)->withQueryString();

        // Get buildings for filter
        $buildings = \App\Models\Building::where('society_id', $society->id)
            ->orderBy('name')
            ->get(['id', 'name']);

        // Get units for filter
        $units = Unit::where('society_id', $society->id)
            ->where('status', 'occupied')
            ->with('building:id,name')
            ->orderBy('unit_no')
            ->get(['id', 'unit_no', 'building_id']);

        return Inertia::render('SocietyAdmin/Directory/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'residents' => $residents,
            'buildings' => $buildings,
            'units' => $units,
            'filters' => $request->only(['search', 'building_id', 'unit_id']),
        ]);
    }
}
