<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Models\Bill;
use App\Models\Expense;
use App\Models\MaintenanceTicket;
use App\Models\Payment;
use App\Models\Society;
use App\Models\Unit;
use App\Models\Visitor;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class DashboardController extends Controller
{
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        // Ensure user is society_admin
        if (! $user->hasRole('society_admin')) {
            abort(403, 'Only Society Admin can access this dashboard.');
        }

        // Ensure user belongs to this society
        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Load subscription with plan relationship
        $society->load('subscription.plan');

        // KPI 1: Total Collections (All time)
        $totalCollections = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->sum('amount');

        // KPI 2: Outstanding Dues
        $outstandingDues = Bill::where('society_id', $society->id)
            ->whereIn('status', ['due', 'overdue'])
            ->sum('amount');

        // KPI 3: This Month Expenses
        $thisMonthExpenses = Expense::where('society_id', $society->id)
            ->whereMonth('expense_date', now()->month)
            ->whereYear('expense_date', now()->year)
            ->sum('amount');

        // KPI 4: Total Units
        $totalUnits = Unit::where('society_id', $society->id)->count();

        // KPI 5: Occupied Units
        $occupiedUnits = Unit::where('society_id', $society->id)
            ->where('status', 'occupied')
            ->count();

        // KPI 6: Open Tickets
        $openTickets = MaintenanceTicket::where('society_id', $society->id)
            ->whereIn('status', ['open', 'assigned', 'in_progress'])
            ->count();

        // KPI 7: Today's Visitors
        $todayVisitors = Visitor::where('society_id', $society->id)
            ->whereDate('created_at', today())
            ->count();

        // KPI 8: Collection Rate
        $totalBills = Bill::where('society_id', $society->id)
            ->whereIn('status', ['due', 'paid', 'overdue'])
            ->sum('amount');
        $paidBills = Bill::where('society_id', $society->id)
            ->where('status', 'paid')
            ->sum('amount');
        $collectionRate = $totalBills > 0 ? round(($paidBills / $totalBills) * 100, 2) : 0;

        // Subscription Plan Status & Expiry
        $subscription = $society->subscription;
        $planStatus = $subscription ? $subscription->status : 'none';
        $planExpiry = $subscription && $subscription->end_date
            ? $subscription->end_date->format('Y-m-d')
            : null;
        $planName = $subscription && $subscription->plan
            ? $subscription->plan->display_name
            : ($subscription ? $subscription->plan_display_name : 'No Plan');

        // Recent Activity Feed
        $recentPayments = Payment::where('society_id', $society->id)
            ->with(['user:id,name', 'bill:id,bill_no'])
            ->latest()
            ->limit(5)
            ->get();

        $recentBills = Bill::where('society_id', $society->id)
            ->with(['user:id,name,unit_id', 'user.unit:id,unit_no'])
            ->latest()
            ->limit(5)
            ->get();

        $recentExpenses = Expense::where('society_id', $society->id)
            ->with(['createdBy:id,name'])
            ->latest()
            ->limit(5)
            ->get();

        // Monthly Collections Chart Data (Last 6 months) - Database agnostic
        $monthlyCollections = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->where('created_at', '>=', now()->subMonths(6))
            ->get()
            ->groupBy(function ($payment) {
                return $payment->created_at->format('Y-m');
            })
            ->map(function ($payments, $month) {
                return [
                    'month' => $month,
                    'total' => (float) $payments->sum('amount'),
                ];
            })
            ->sortBy('month')
            ->values();

        // Monthly Expenses Chart Data (Last 6 months) - Database agnostic
        $monthlyExpenses = Expense::where('society_id', $society->id)
            ->where('expense_date', '>=', now()->subMonths(6)->startOfMonth())
            ->get()
            ->groupBy(function ($expense) {
                return $expense->expense_date->format('Y-m');
            })
            ->map(function ($expenses, $month) {
                return [
                    'month' => $month,
                    'total' => (float) $expenses->sum('amount'),
                ];
            })
            ->sortBy('month')
            ->values();



        // KPI: Ad Revenue
        $adRevenue = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->whereNull('bill_id')
            ->sum('amount');

        return Inertia::render('SocietyAdmin/Dashboard', [
            'society' => $society->only(['id', 'name', 'code', 'registration_no']),
            'kpis' => [
                'total_collections' => (float) $totalCollections,
                'outstanding_dues' => (float) $outstandingDues,
                'this_month_expenses' => (float) $thisMonthExpenses,
                'total_units' => $totalUnits,
                'occupied_units' => $occupiedUnits,
                'open_tickets' => $openTickets,
                'today_visitors' => $todayVisitors,
                'collection_rate' => $collectionRate,
                'plan_status' => $planStatus,
                'plan_expiry' => $planExpiry,

                'plan_name' => $planName,
                'ad_revenue' => (float) $adRevenue,
            ],
            'recentActivity' => [
                'payments' => $recentPayments->map(function ($payment) {
                    return [
                        'id' => $payment->id,
                        'payment_no' => $payment->payment_no,
                        'amount' => (float) $payment->amount,
                        'user_name' => $payment->user->name ?? 'N/A',
                        'bill_no' => $payment->bill->bill_no ?? null,
                        'created_at' => $payment->created_at->format('Y-m-d H:i'),
                    ];
                }),
                'bills' => $recentBills->map(function ($bill) {
                    return [
                        'id' => $bill->id,
                        'bill_no' => $bill->bill_no,
                        'amount' => (float) $bill->amount,
                        'status' => $bill->status,
                        'user_name' => $bill->user->name ?? 'N/A',
                        'unit_no' => $bill->user->unit->unit_no ?? 'N/A',
                        'created_at' => $bill->created_at->format('Y-m-d H:i'),
                    ];
                }),
                'expenses' => $recentExpenses->map(function ($expense) {
                    return [
                        'id' => $expense->id,
                        'expense_no' => $expense->expense_no,
                        'amount' => (float) $expense->amount,
                        'category' => $expense->category,
                        'created_by_name' => $expense->createdBy->name ?? 'N/A',
                        'created_at' => $expense->created_at->format('Y-m-d H:i'),
                    ];
                }),
            ],
            'charts' => [
                'monthly_collections' => $monthlyCollections,
                'monthly_expenses' => $monthlyExpenses,
            ],
        ]);
    }
}
