<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Models\CollectorTransaction;
use App\Models\CollectorWallet;
use App\Models\Payment;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Inertia\Response;

class CollectorWalletController extends Controller
{
    /**
     * List all collectors with their wallet balances.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        // Ensure user belongs to this society
        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Get all collectors for this society
        $collectors = User::where('society_id', $society->id)
            ->whereHas('role', function ($q) {
                $q->where('name', 'collector');
            })
            ->get()
            ->map(function ($collector) use ($society) {
                // Get or create wallet and sync with actual payments
                $wallet = CollectorWallet::getOrCreate($collector->id, $society->id);
                
                // Calculate actual cash collected from payments
                $actualCashCollected = Payment::where('society_id', $society->id)
                    ->where('collected_by', $collector->id)
                    ->where('status', 'completed')
                    ->where('payment_method', 'cash')
                    ->sum('amount');
                
                $actualCashHandover = $wallet->total_handover ?? 0;
                $expectedBalance = max(0, $actualCashCollected - $actualCashHandover);
                
                // Sync wallet if it's out of sync
                if ($wallet->total_collected < $actualCashCollected) {
                    // Some payments weren't credited - sync the wallet
                    $difference = $actualCashCollected - $wallet->total_collected;
                    $wallet->balance += $difference;
                    $wallet->total_collected = $actualCashCollected;
                    $wallet->save();
                } elseif ($wallet->transactions()->count() == 0 && $expectedBalance != $wallet->balance) {
                    // No transactions yet, but wallet is out of sync - sync it
                    $wallet->balance = $expectedBalance;
                    $wallet->total_collected = $actualCashCollected;
                    $wallet->save();
                }
                
                // Refresh wallet to get latest values
                $wallet->refresh();
                
                return [
                    'id' => $collector->id,
                    'name' => $collector->name,
                    'email' => $collector->email,
                    'phone' => $collector->phone,
                    'wallet' => [
                        'id' => $wallet->id,
                        'balance' => (float) $wallet->balance,
                        'total_collected' => (float) $wallet->total_collected,
                        'total_handover' => (float) $wallet->total_handover,
                    ],
                ];
            });

        // Summary statistics
        $totalCashInHand = CollectorWallet::where('society_id', $society->id)->sum('balance');
        $totalCollected = CollectorWallet::where('society_id', $society->id)->sum('total_collected');
        $totalHandover = CollectorWallet::where('society_id', $society->id)->sum('total_handover');

        return Inertia::render('SocietyAdmin/Collectors/Wallets', [
            'society' => $society->only(['id', 'name', 'code']),
            'collectors' => $collectors,
            'summary' => [
                'total_cash_in_hand' => (float) $totalCashInHand,
                'total_collected' => (float) $totalCollected,
                'total_handover' => (float) $totalHandover,
                'collectors_count' => $collectors->count(),
            ],
        ]);
    }

    /**
     * Show collector wallet details with transactions.
     */
    public function show(Request $request, Society $society, int $collectorId): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $collector = User::where('society_id', $society->id)
            ->where('id', $collectorId)
            ->whereHas('role', function ($q) {
                $q->where('name', 'collector');
            })
            ->firstOrFail();

        $wallet = CollectorWallet::getOrCreate($collectorId, $society->id);
        
        // Sync wallet with actual payments
        $actualCashCollected = Payment::where('society_id', $society->id)
            ->where('collected_by', $collectorId)
            ->where('status', 'completed')
            ->where('payment_method', 'cash')
            ->sum('amount');
        
        $actualCashHandover = $wallet->total_handover ?? 0;
        $expectedBalance = max(0, $actualCashCollected - $actualCashHandover);
        
        // Sync wallet if it's out of sync
        if ($wallet->total_collected < $actualCashCollected) {
            $difference = $actualCashCollected - $wallet->total_collected;
            $wallet->balance += $difference;
            $wallet->total_collected = $actualCashCollected;
            $wallet->save();
        } elseif ($wallet->transactions()->count() == 0 && $expectedBalance != $wallet->balance) {
            $wallet->balance = $expectedBalance;
            $wallet->total_collected = $actualCashCollected;
            $wallet->save();
        }
        
        $wallet->refresh();

        // Create missing transactions for existing payments that don't have transaction records
        $existingTransactionPaymentIds = CollectorTransaction::where('wallet_id', $wallet->id)
            ->where('reference_type', 'payment')
            ->whereNotNull('reference_id')
            ->pluck('reference_id')
            ->toArray();
        
        $paymentsWithoutTransactions = Payment::where('society_id', $society->id)
            ->where('collected_by', $collectorId)
            ->where('status', 'completed')
            ->where('payment_method', 'cash')
            ->whereNotIn('id', $existingTransactionPaymentIds)
            ->orderBy('created_at', 'asc')
            ->get();
        
        if ($paymentsWithoutTransactions->count() > 0) {
            // Get all existing transactions to calculate running balance
            $existingTransactions = CollectorTransaction::where('wallet_id', $wallet->id)
                ->orderBy('created_at', 'asc')
                ->get();
            
            // Calculate initial balance (before any transactions)
            $runningBalance = 0;
            
            // Process existing transactions to get current running balance
            foreach ($existingTransactions as $txn) {
                if ($txn->type === 'credit') {
                    $runningBalance += $txn->amount;
                } elseif ($txn->type === 'debit') {
                    $runningBalance -= $txn->amount;
                }
            }
            
            // Now create missing transactions for payments
            foreach ($paymentsWithoutTransactions as $payment) {
                $runningBalance += $payment->amount;
                
                // Create transaction record
                CollectorTransaction::create([
                    'wallet_id' => $wallet->id,
                    'collector_id' => $collectorId,
                    'society_id' => $society->id,
                    'type' => 'credit',
                    'amount' => $payment->amount,
                    'balance_after' => $runningBalance,
                    'reference_type' => 'payment',
                    'reference_id' => $payment->id,
                    'notes' => 'Cash payment collected - Payment #' . $payment->payment_no,
                    'created_at' => $payment->created_at,
                    'updated_at' => $payment->updated_at,
                ]);
            }
            
            // Refresh wallet after creating missing transactions
            $wallet->refresh();
        }

        // Get transactions with pagination
        $transactions = $wallet->transactions()
            ->with('receiver:id,name')
            ->orderBy('created_at', 'desc')
            ->paginate(20)
            ->through(function ($txn) {
                return [
                    'id' => $txn->id,
                    'type' => $txn->type,
                    'amount' => (float) $txn->amount,
                    'balance_after' => (float) $txn->balance_after,
                    'reference_type' => $txn->reference_type,
                    'notes' => $txn->notes,
                    'received_by' => $txn->receiver?->name,
                    'created_at' => $txn->created_at->format('Y-m-d H:i:s'),
                ];
            });

        return Inertia::render('SocietyAdmin/Collectors/WalletDetails', [
            'society' => $society->only(['id', 'name', 'code']),
            'collector' => [
                'id' => $collector->id,
                'name' => $collector->name,
                'email' => $collector->email,
                'phone' => $collector->phone,
            ],
            'wallet' => [
                'id' => $wallet->id,
                'balance' => (float) $wallet->balance,
                'total_collected' => (float) $wallet->total_collected,
                'total_handover' => (float) $wallet->total_handover,
            ],
            'transactions' => $transactions,
        ]);
    }

    /**
     * Record a cash handover from collector (admin initiated).
     */
    public function recordHandover(Request $request, Society $society, int $collectorId)
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'notes' => 'nullable|string|max:500',
        ]);

        $collector = User::where('society_id', $society->id)
            ->where('id', $collectorId)
            ->whereHas('role', function ($q) {
                $q->where('name', 'collector');
            })
            ->firstOrFail();

        $wallet = CollectorWallet::getOrCreate($collectorId, $society->id);
        $amount = $request->get('amount');

        if ($amount > $wallet->balance) {
            return back()->withErrors([
                'amount' => 'Amount exceeds collector wallet balance (₹' . number_format($wallet->balance, 2) . ')',
            ]);
        }

        try {
            DB::transaction(function () use ($wallet, $amount, $user, $request, $society) {
                // Debit collector's wallet
                $wallet->debit($amount, $user->id, $request->get('notes') ?? 'Received by ' . $user->name);

                // Record bank transaction
                $primaryAccount = \App\Models\BankAccount::where('society_id', $society->id)
                    ->where('is_primary', true)
                    ->first();
                    
                if (!$primaryAccount) {
                    $primaryAccount = \App\Models\BankAccount::where('society_id', $society->id)->first();
                }

                if ($primaryAccount) {
                    app(\App\Services\TransactionService::class)->handleCashHandover(
                        $wallet->collector_id,
                        $primaryAccount->id,
                        $amount,
                        $request->get('notes')
                    );
                }
            });

            return back()->with('success', 'Cash handover recorded successfully. Amount: ₹' . number_format($amount, 2));
        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Failed to record handover: ' . $e->getMessage()]);
        }
    }
}
