<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreBuildingRequest;
use App\Http\Requests\SocietyAdmin\UpdateBuildingRequest;
use App\Models\Building;
use App\Models\Society;
use App\Models\Unit;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class BuildingController extends Controller
{
    /**
     * Display a listing of buildings.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        // Ensure user belongs to this society
        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $query = Building::where('society_id', $society->id)
            ->withCount('units');

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('building_no', 'like', "%{$search}%");
            });
        }

        $buildings = $query->latest()->paginate(15)->withQueryString();

        // Calculate Stats
        $totalBuildings = Building::where('society_id', $society->id)->count();
        $totalFloors = Building::where('society_id', $society->id)->sum('floors');
        $totalUnits = Unit::where('society_id', $society->id)->count();

        return Inertia::render('SocietyAdmin/Buildings/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'buildings' => $buildings,
            'filters' => $request->only(['search']),
            'stats' => [
                'total_buildings' => $totalBuildings,
                'total_floors' => (int) $totalFloors,
                'total_units' => $totalUnits,
            ],
        ]);
    }

    /**
     * Show the form for creating a new building.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        return Inertia::render('SocietyAdmin/Buildings/Create', [
            'society' => $society->only(['id', 'name', 'code']),
        ]);
    }

    /**
     * Store a newly created building.
     */
    public function store(StoreBuildingRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;

        Building::create($data);

        return redirect()->route('society.buildings.index', $society)
            ->with('success', 'Building created successfully.');
    }

    /**
     * Display the specified building.
     */
    public function show(Society $society, Building $building): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $building->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $building->load(['units.user:id,name,phone', 'society:id,name']);

        return Inertia::render('SocietyAdmin/Buildings/Show', [
            'society' => $society->only(['id', 'name', 'code']),
            'building' => $building,
        ]);
    }

    /**
     * Show the form for editing the specified building.
     */
    public function edit(Society $society, Building $building): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $building->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        return Inertia::render('SocietyAdmin/Buildings/Edit', [
            'society' => $society->only(['id', 'name', 'code']),
            'building' => $building,
        ]);
    }

    /**
     * Update the specified building.
     */
    public function update(UpdateBuildingRequest $request, Society $society, Building $building): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $building->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $building->update($request->validated());

        return redirect()->route('society.buildings.index', $society)
            ->with('success', 'Building updated successfully.');
    }

    /**
     * Remove the specified building.
     */
    public function destroy(Society $society, Building $building): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $building->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Check if building has units
        if ($building->units()->count() > 0) {
            return redirect()->back()
                ->with('error', 'Cannot delete building with existing units. Please remove all units first.');
        }

        $building->delete();

        return redirect()->route('society.buildings.index', $society)
            ->with('success', 'Building deleted successfully.');
    }
}
