<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreBillCycleRequest;
use App\Models\Bill;
use App\Models\BillCycle;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use Inertia\Response;

class BillCycleController extends Controller
{
    /**
     * Display a listing of bill cycles.
     */
    public function index(Request $request, Society $society): Response
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $query = BillCycle::where('society_id', $society->id)
            ->withCount('bills');

        // Filter by status
        if ($request->has('status') && $request->get('status') !== 'all') {
            $query->where('status', $request->get('status'));
        }

        // Filter by year
        if ($request->has('year') && $request->get('year')) {
            $query->where('year', $request->get('year'));
        }

        $billCycles = $query->latest()->paginate(10)->withQueryString();

        // Calculate stats
        $stats = [
            'total' => BillCycle::where('society_id', $society->id)->count(),
            'draft' => BillCycle::where('society_id', $society->id)->where('status', 'draft')->count(),
            'under_review' => BillCycle::where('society_id', $society->id)->where('status', 'generated')->count(),
            'posted' => BillCycle::where('society_id', $society->id)->where('status', 'posted')->count(),
            'closed' => BillCycle::where('society_id', $society->id)->where('status', 'closed')->count(),
        ];

        $maintenanceStructures = \App\Models\MaintenanceStructure::where('society_id', $society->id)
            ->get(['id', 'name', 'maintenance', 'parking', 'other_charges']);

        return Inertia::render('SocietyAdmin/BillCycles/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'billCycles' => $billCycles,
            'filters' => $request->only(['status', 'year']),
            'stats' => $stats,
            'maintenanceStructures' => $maintenanceStructures,
        ]);
    }

    /**
     * Show the form for creating a new bill cycle.
     */
    public function create(Society $society): Response
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Sub-Admin: No bill cycle creation
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin cannot create bill cycles.');
        }

        return Inertia::render('SocietyAdmin/BillCycles/Create', [
            'society' => $society->only(['id', 'name', 'code']),
        ]);
    }

    /**
     * Store a newly created bill cycle.
     */
    public function store(StoreBillCycleRequest $request, Society $society): RedirectResponse
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Sub-Admin: No bill cycle creation
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin cannot create bill cycles.');
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;

        BillCycle::create($data);

        return redirect()->route('society.bill-cycles.index', $society)
            ->with('success', 'Bill cycle created successfully.');
    }

    /**
     * Generate bills for all residents in a bill cycle.
     */
    public function generateBills(Request $request, Society $society, BillCycle $billCycle): RedirectResponse
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        if ($user->society_id !== $society->id || $billCycle->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No bulk bill generation
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin cannot generate bills in bulk.');
        }

        // Allow regeneration - removed the check for 'generated' status
        // Users can now regenerate bills to update amounts based on new maintenance structures

        if ($billCycle->status === 'closed') {
            return redirect()->back()
                ->with('error', 'This bill cycle is closed and cannot generate bills.');
        }

        $residentRole = Role::where('name', 'resident')->first();

        if (! $residentRole) {
            return redirect()->back()
                ->with('error', 'Resident role not found.');
        }

        // Get all active residents with units (exclude family members and tenants - only owners receive bills)
        $residents = User::where('society_id', $society->id)
            ->where('role_id', $residentRole->id)
            ->where('status', 'active')
            ->whereNotNull('unit_id')
            ->where(function ($query) {
                // Only include owners (exclude family_member and tenant)
                $query->where('type', 'owner')
                      ->orWhereNull('type'); // Also include null type for backward compatibility
            })
            ->with(['unit.maintenanceStructure'])
            ->get();

        if ($residents->isEmpty()) {
            return redirect()->back()
                ->with('error', 'No active residents with units found to generate bills.');
        }

        $action = $request->input('action', 'generate');
        $manualStructureId = $request->input('maintenance_structure_id');
        $manualStructure = $manualStructureId 
            ? \App\Models\MaintenanceStructure::find($manualStructureId) 
            : null;

        // Get default maintenance structure for the society (first one or fallback)
        $defaultStructure = \App\Models\MaintenanceStructure::where('society_id', $society->id)
            ->orderBy('id')
            ->first();

        // Calculate due date: 5 days before end of month (grace period)
        $billCycleDate = \Carbon\Carbon::parse($billCycle->bill_date ?? $billCycle->created_at);
        $lastDayOfMonth = $billCycleDate->copy()->endOfMonth();
        $graceDays = 5; // 5 days grace period before end of month
        $calculatedDueDate = $lastDayOfMonth->copy()->subDays($graceDays); // 5 days before month end
        
        // Use calculated due date or fallback to bill cycle due_date if it exists
        $dueDate = $billCycle->due_date ?? $calculatedDueDate->format('Y-m-d');

        DB::transaction(function () use ($residents, $billCycle, $society, $user, $action, $manualStructure, $defaultStructure, $dueDate) {
            // Get the next bill number for this society (standardized format)
            $prefix = 'BILL-SOC-' . date('Y') . '-';
            
            $lastBill = Bill::where('society_id', $society->id)
                ->where('bill_no', 'like', $prefix . '%')
                ->orderBy('id', 'desc')
                ->first();

            $sequence = 0;
            if ($lastBill) {
                $parts = explode('-', $lastBill->bill_no);
                $lastSeq = end($parts);
                if (is_numeric($lastSeq)) {
                    $sequence = intval($lastSeq);
                }
            }

            // Start sequence for this batch
            $currentSequence = $sequence + 1;

            foreach ($residents as $resident) {
                // Determine Structure to use
                // Priority 1: Manual Override (if selected in modal)
                // Priority 2: Unit's Assigned Structure
                // Priority 3: Default Structure for Society
                // Priority 4: Fallback to 0 (only if no structures exist)
                
                $structureToUse = $manualStructure;
                if (! $structureToUse && $resident->unit && $resident->unit->maintenanceStructure) {
                    $structureToUse = $resident->unit->maintenanceStructure;
                }
                if (! $structureToUse && $defaultStructure) {
                    $structureToUse = $defaultStructure;
                }

                // Calculate amounts using structure or fallback to 0
                $maintenance = $structureToUse ? $structureToUse->maintenance : 0;
                $parking = $structureToUse ? $structureToUse->parking : 0;
                $otherCharges = $structureToUse ? $structureToUse->other_charges : 0;
                
                $water = 0; // Variable
                $electricity = 0; // Variable
                $gst = 0;
                $discount = 0;

                // Check if bill already exists for this cycle and resident
                $existingBill = Bill::where('bill_cycle_id', $billCycle->id)
                    ->where('user_id', $resident->id)
                    ->first();

                if ($existingBill) {
                    if ($action === 'regenerate') {
                        // Regenerate: Update existing bill with new structure values (for any status except paid/cancelled)
                        if (!in_array($existingBill->status, ['paid', 'cancelled'])) {
                            $newAmount = $maintenance + $parking + $otherCharges + $existingBill->water + $existingBill->electricity + $existingBill->gst - $existingBill->discount;
                            
                            $existingBill->update([
                                'maintenance' => $maintenance,
                                'parking' => $parking,
                                'other_charges' => $otherCharges,
                                'amount' => $newAmount,
                                'due_date' => $dueDate,
                                // Don't change status if already approved or due
                                'status' => in_array($existingBill->status, ['approved', 'due']) ? $existingBill->status : 'under_review',
                            ]);
                        }
                    }
                    continue; 
                }

                // Generate standardized bill number
                $billNo = $prefix . str_pad($currentSequence++, 6, '0', STR_PAD_LEFT);

                $amount = $maintenance + $water + $electricity + $parking + $otherCharges + $gst - $discount;

                // Determine initial status
                // If amount is 0, set to 'draft' (not paid, needs review)
                // Otherwise, set to 'under_review' (will be approved when posted)
                $initialStatus = ($amount == 0) ? 'draft' : 'under_review';

                Bill::create([
                    'society_id' => $society->id,
                    'bill_cycle_id' => $billCycle->id,
                    'user_id' => $resident->id,
                    'bill_no' => $billNo,
                    'amount' => $amount,
                    'maintenance' => $maintenance,
                    'water' => $water,
                    'electricity' => $electricity,
                    'parking' => $parking,
                    'other_charges' => $otherCharges,
                    'gst' => $gst,
                    'discount' => $discount,
                    'status' => $initialStatus,
                    'due_date' => $dueDate,
                    'generated_by' => $user->id,
                ]);
            }

            // Update bill cycle status
            $billCycle->update(['status' => 'generated']);
        });

        $message = $action === 'regenerate' 
            ? 'Bills regenerated successfully (only Under Review bills were updated).' 
            : 'Bills generated successfully and are under review.';

        return redirect()->route('society.bill-cycles.index', $society)
            ->with('success', $message);
    }

    /**
     * Post bills to residents (make them visible and notify).
     */
    public function postBills(Society $society, BillCycle $billCycle): RedirectResponse
    {
        /** @var \App\Models\User|null $authUser */
        $authUser = Auth::user();
        
        if (!$authUser) {
            abort(401, 'Unauthenticated.');
        }
        
        $user = $authUser;

        if ($user->society_id !== $society->id || $billCycle->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No bill posting
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin cannot post bills.');
        }

        if ($billCycle->status === 'closed') {
            return redirect()->back()
                ->with('error', 'This bill cycle is closed.');
        }

        if ($billCycle->status === 'posted') {
            return redirect()->back()
                ->with('error', 'Bills have already been posted for this cycle.');
        }

        DB::transaction(function () use ($billCycle) {
            // Update all under_review and draft bills to approved status when posting
            // Skip bills with amount 0 (draft) - they need manual review
            Bill::where('bill_cycle_id', $billCycle->id)
                ->whereIn('status', ['under_review', 'draft'])
                ->where('amount', '>', 0) // Only approve bills with amount > 0
                ->update(['status' => 'approved']);

            // Update bill cycle status to posted
            $billCycle->update(['status' => 'posted']);
        });

        // TODO: Send notifications to residents (email/push)

        return redirect()->route('society.bill-cycles.index', $society)
            ->with('success', 'Bills have been posted successfully and are now visible to residents.');
    }
}
