<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreBillRequest;
use App\Http\Requests\SocietyAdmin\UpdateBillRequest;
use App\Models\Bill;
use App\Models\BillCycle;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class BillController extends Controller
{
    /**
     * Display a listing of bills.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $query = Bill::where('society_id', $society->id)
            ->with(['user:id,name,phone,email,unit_id', 'user.unit:id,unit_no', 'billCycle:id,name', 'generatedBy:id,name', 'payment']);

        // Residents can only see their own bills that are NOT under review
        if ($user->hasRole('resident')) {
            $query->where('user_id', $user->id)
                ->where('status', '!=', 'under_review');
        }

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('bill_no', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%");
                    });
            });
        }

        // Filter by status
        if ($request->has('status') && $request->get('status') !== 'all') {
            $query->where('status', $request->get('status'));
        }

        // Filter by bill cycle
        if ($request->has('bill_cycle_id') && $request->get('bill_cycle_id') !== 'all') {
            $query->where('bill_cycle_id', $request->get('bill_cycle_id'));
        }

        // Filter by resident
        if ($request->has('user_id') && $request->get('user_id') !== 'all') {
            $query->where('user_id', $request->get('user_id'));
        }

        $bills = $query->latest()->paginate(15)->withQueryString();

        // Calculate stats
        $stats = [
            'total_bills' => Bill::where('society_id', $society->id)->count(),
            'pending_amount' => Bill::where('society_id', $society->id)->whereIn('status', ['due', 'overdue'])->sum('amount'),
            'collected_amount' => Bill::where('society_id', $society->id)->where('status', 'paid')->sum('amount'),
            'under_review' => Bill::where('society_id', $society->id)->where('status', 'under_review')->count(),
        ];

        // Get bill cycles for filter
        $billCycles = BillCycle::where('society_id', $society->id)
            ->orderBy('name', 'desc')
            ->get(['id', 'name']);

        // Get residents for filter
        $residentRole = Role::where('name', 'resident')->first();
        $residents = $residentRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $residentRole->id)
                ->orderBy('name')
                ->get(['id', 'name', 'phone'])
            : collect();

        // Get maintenance structures
        $maintenanceStructures = \App\Models\MaintenanceStructure::where('society_id', $society->id)
            ->get(['id', 'name', 'maintenance', 'parking', 'other_charges']);

        return Inertia::render('SocietyAdmin/Bills/Index', [
            'society' => $society->only([
                'id', 'name', 'code', 'address', 'city', 'state', 'pincode', 
                'phone', 'email', 'gst_no', 'registration_no', 'logo'
            ]),
            'bills' => $bills,
            'billCycles' => $billCycles,
            'residents' => $residents,
            'maintenanceStructures' => $maintenanceStructures,
            'filters' => $request->only(['search', 'status', 'bill_cycle_id', 'user_id']),
            'stats' => $stats,
        ]);
    }

    /**
     * Show the form for creating a new bill.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Get bill cycles
        $billCycles = BillCycle::where('society_id', $society->id)
            ->where('status', '!=', 'closed')
            ->orderBy('name', 'desc')
            ->get(['id', 'name']);

        // Get residents
        $residentRole = Role::where('name', 'resident')->first();
        $residents = $residentRole
            ? User::where('society_id', $society->id)
                ->where('role_id', $residentRole->id)
                ->where('status', 'active')
                ->with('unit:id,unit_no')
                ->orderBy('name')
                ->get(['id', 'name', 'phone', 'unit_id'])
            : collect();

        return Inertia::render('SocietyAdmin/Bills/Create', [
            'society' => $society->only(['id', 'name', 'code']),
            'billCycles' => $billCycles,
            'residents' => $residents,
        ]);
    }

    /**
     * Store a newly created bill.
     */
    public function store(StoreBillRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;
        $data['generated_by'] = $user->id;

        // Convert "none" values to null for optional fields
        if (isset($data['bill_cycle_id']) && $data['bill_cycle_id'] === 'none') {
            $data['bill_cycle_id'] = null;
        }

        if (!isset($data['status'])) {
             $data['status'] = 'due'; // Default to due for manual creation
        }

        $residents = collect();

        if ($data['user_id'] === 'all') {
            $residentRole = Role::where('name', 'resident')->first();
            if ($residentRole) {
                $residents = User::where('society_id', $society->id)
                    ->where('role_id', $residentRole->id)
                    ->where('status', 'active')
                    ->get();
            }
        } else {
            $residents->push(User::find($data['user_id']));
        }

        $count = 0;
        foreach ($residents as $resident) {
            $billData = $data;
            $billData['user_id'] = $resident->id;

            // Generate sequential bill number
            $prefix = 'BILL-SOC-' . date('Y') . '-';
            
            $lastBill = Bill::where('society_id', $society->id)
                ->where('bill_no', 'like', $prefix . '%')
                ->orderBy('id', 'desc')
                ->first();

            $sequence = 0;
            if ($lastBill) {
                // Extract sequence number from the end
                $parts = explode('-', $lastBill->bill_no);
                $lastSeq = end($parts);
                if (is_numeric($lastSeq)) {
                    $sequence = intval($lastSeq);
                }
            }

            $billNo = $prefix . str_pad($sequence + 1, 6, '0', STR_PAD_LEFT);
            $billData['bill_no'] = $billNo;

            // Calculate total amount
            $billData['amount'] = ($billData['maintenance'] ?? 0) +
                ($billData['water'] ?? 0) +
                ($billData['electricity'] ?? 0) +
                ($billData['parking'] ?? 0) +
                ($billData['other_charges'] ?? 0) +
                ($billData['gst'] ?? 0) -
                ($billData['discount'] ?? 0);

            $bill = Bill::create($billData);
            $count++;

            // Trigger Notification
            // $resident->notify(new \App\Notifications\BillGenerated($bill)); // Keep existing mail/db notification if needed

            // Send Push Notification
            if ($bill->status === 'due' && $resident->fcm_token) {
                try {
                     $firebaseService = app(\App\Services\FirebaseNotificationService::class);
                     $firebaseService->sendToUser(
                        $resident,
                        'New Bill Generated',
                        "Bill #{$bill->bill_no} for period " . ($bill->billCycle ? $bill->billCycle->name : 'N/A') . " generated. Amount: ₹{$bill->amount}",
                        [
                            'type' => 'new_bill',
                            'bill_id' => (string) $bill->id,
                        ]
                    );
                } catch (\Exception $e) {
                    // Log error for debugging notification failures (non-critical)
                    \Illuminate\Support\Facades\Log::error('Failed to send bill notification: ' . $e->getMessage());
                }
            }
        }

        return redirect()->route('society.bills.index', $society)
            ->with('success', "{$count} bill(s) created successfully.");
    }

    /**
     * Display the specified bill.
     */
    public function show(Society $society, Bill $bill): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $bill->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: View only, no show page
        if ($user->hasRole('sub_admin') && ! $user->hasPermission('billing.bills.view')) {
            abort(403, 'You do not have permission to view bill details.');
        }

        $bill->load([
            'user:id,name,phone,email,unit_id',
            'user.unit:id,unit_no,building_id',
            'user.unit.building:id,name',
            'billCycle:id,name',
            'generatedBy:id,name',
            'revisedBy:id,name',
            'payment',
        ]);

        return Inertia::render('SocietyAdmin/Bills/Show', [
            'society' => $society->only(['id', 'name', 'code']),
            'bill' => $bill,
        ]);
    }

    /**
     * Update the specified bill.
     */
    public function update(UpdateBillRequest $request, Society $society, Bill $bill): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $bill->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No update permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only view and create bills.');
        }

        if ($bill->status === 'paid') {
            return redirect()->back()
                ->with('error', 'Cannot update a paid bill.');
        }

        $data = $request->validated();

        // Calculate new total amount
        $data['amount'] = ($data['maintenance'] ?? 0) +
            ($data['water'] ?? 0) +
            ($data['electricity'] ?? 0) +
            ($data['parking'] ?? 0) +
            ($data['other_charges'] ?? 0) +
            ($data['gst'] ?? 0) -
            ($data['discount'] ?? 0);

        // Track revision if it was already due/sent, but for under_review it's just an update
        if ($bill->status !== 'under_review') {
             $data['revised_by'] = $user->id;
             $data['revised_at'] = now();
        }

        $bill->update($data);

        return redirect()->back()
            ->with('success', 'Bill updated successfully.');
    }

    /**
     * Approve a bill or multiple bills.
     */
    public function approve(Request $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin check
        if ($user->hasRole('sub_admin')) {
             abort(403, 'Sub-Admin cannot approve bills.');
        }

        $request->validate([
            'bill_ids' => ['required', 'array'],
            'bill_ids.*' => ['exists:bills,id'],
        ]);

        $bills = Bill::whereIn('id', $request->bill_ids)
            ->where('society_id', $society->id)
            ->where('status', 'under_review')
            ->with('user')
            ->get();

        $count = 0;
        $firebaseService = app(\App\Services\FirebaseNotificationService::class);

        foreach ($bills as $bill) {
            $bill->update(['status' => 'due']);
            $count++;

            if ($bill->user && $bill->user->fcm_token) {
                 try {
                     $firebaseService->sendToUser(
                        $bill->user,
                        'Bill Approved',
                        "Your bill #{$bill->bill_no} is now approved and due for payment. Amount: ₹{$bill->amount}",
                        [
                            'type' => 'new_bill',
                            'bill_id' => (string) $bill->id,
                        ]
                    );
                } catch (\Exception $e) {
                    // Log error for debugging notification failures (non-critical)
                    \Illuminate\Support\Facades\Log::error('Failed to send bill approval notification: ' . $e->getMessage());
                }
            }
        }

        return redirect()->back()
            ->with('success', "{$count} bills approved successfully.");
    }
}
