<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreBankAccountRequest;
use App\Http\Requests\SocietyAdmin\UpdateBankAccountRequest;
use App\Models\BankAccount;
use App\Models\Payment;
use App\Models\Society;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class BankAccountController extends Controller
{
    /**
     * Display a listing of bank accounts.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $query = BankAccount::where('society_id', $society->id)
            ->withCount('payments')
            ->with('updatedBy:id,name');

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('account_name', 'like', "%{$search}%")
                    ->orWhere('account_number', 'like', "%{$search}%")
                    ->orWhere('bank_name', 'like', "%{$search}%")
                    ->orWhere('ifsc_code', 'like', "%{$search}%");
            });
        }

        // Filter by account type
        if ($request->has('account_type') && $request->get('account_type') !== 'all') {
            $query->where('account_type', $request->get('account_type'));
        }

        $bankAccounts = $query->latest()->paginate(15)->withQueryString();

        // Calculate stats
        $stats = [
            'total_balance' => BankAccount::where('society_id', $society->id)->sum('balance'),
            'total_accounts' => BankAccount::where('society_id', $society->id)->count(),
            'total_transactions' => Payment::whereHas('bankAccount', function ($q) use ($society) {
                $q->where('society_id', $society->id);
            })->count(),
            'savings_balance' => BankAccount::where('society_id', $society->id)
                ->where('account_type', 'savings')
                ->sum('balance'),
        ];

        return Inertia::render('SocietyAdmin/BankAccounts/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'bankAccounts' => $bankAccounts,
            'filters' => $request->only(['search', 'account_type']),
            'stats' => $stats,
        ]);
    }

    /**
     * Show the form for creating a new bank account.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        return Inertia::render('SocietyAdmin/BankAccounts/Create', [
            'society' => $society->only(['id', 'name', 'code']),
        ]);
    }

    /**
     * Store a newly created bank account.
     */
    public function store(StoreBankAccountRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;
        $data['updated_by'] = $user->id;
        // Default is_primary to false if not provided, but handled by boolean rule/HTML input usually
        $data['is_primary'] = $data['is_primary'] ?? false;

        $bankAccount = BankAccount::create($data);

        if ($data['is_primary']) {
            $bankAccount->markAsPrimary();
        }

        return redirect()->route('society.bank-accounts.index', $society)
            ->with('success', 'Bank account created successfully.');
    }

    /**
     * Show the form for editing the specified bank account.
     */
    public function edit(Society $society, BankAccount $bankAccount): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $bankAccount->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $bankAccount->load('updatedBy:id,name');

        return Inertia::render('SocietyAdmin/BankAccounts/Edit', [
            'society' => $society->only(['id', 'name', 'code']),
            'bankAccount' => $bankAccount,
        ]);
    }

    /**
     * Update the specified bank account.
     */
    public function update(UpdateBankAccountRequest $request, Society $society, BankAccount $bankAccount): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $bankAccount->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $data = $request->validated();
        $data['updated_by'] = $user->id;
        $data['is_primary'] = $data['is_primary'] ?? false;

        $bankAccount->update($data);

        if ($data['is_primary']) {
            $bankAccount->markAsPrimary();
        }

        return redirect()->route('society.bank-accounts.index', $society)
            ->with('success', 'Bank account updated successfully.');
    }

    /**
     * Set the bank account as primary.
     */
    public function setPrimary(Society $society, BankAccount $bankAccount): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $bankAccount->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $bankAccount->markAsPrimary();

        return redirect()->route('society.bank-accounts.index', $society)
            ->with('success', 'Bank account marked as primary.');
    }

    /**
     * Update the balance of the specified bank account.
     */
    public function updateBalance(Request $request, Society $society, BankAccount $bankAccount): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $bankAccount->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $request->validate([
            'balance' => ['required', 'numeric', 'min:0', 'max:999999999999.99'],
        ]);

        $bankAccount->update([
            'balance' => $request->get('balance'),
            'updated_by' => $user->id,
        ]);

        return redirect()->route('society.bank-accounts.index', $society)
            ->with('success', 'Bank account balance updated successfully.');
    }

    /**
     * Display transaction history for the bank account (Bank Statement).
     */
    public function transactions(Request $request, Society $society, BankAccount $bankAccount): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $bankAccount->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $query = \App\Models\Transaction::where('bank_account_id', $bankAccount->id)
            // Exclude transactions linked to failed or pending payments
            ->whereNotExists(function ($query) {
                $query->select(\DB::raw(1))
                    ->from('payments')
                    ->whereColumn('payments.id', 'transactions.reference_id')
                    ->where('transactions.reference_type', 'App\\Models\\Payment')
                    ->whereIn('payments.status', ['failed', 'pending']);
            });

        // Filter by date
        if ($request->has('start_date') && $request->get('start_date')) {
            $query->whereDate('transaction_date', '>=', $request->get('start_date'));
        }

        if ($request->has('end_date') && $request->get('end_date')) {
            $query->whereDate('transaction_date', '<=', $request->get('end_date'));
        }

        $transactions = $query->with(['createdBy:id,name', 'reference'])
            ->latest('transaction_date')
            ->latest('id')
            ->paginate(15)
            ->withQueryString()
            ->through(function ($tx) {
                return [
                    'id' => $tx->id,
                    'type' => $tx->type,
                    'date' => $tx->transaction_date->format('Y-m-d H:i:s'),
                    'description' => $tx->description,
                    'reference' => $this->formatReference($tx->reference),
                    'payment_method' => $tx->payment_method ?? 'N/A',
                    'debit' => $tx->type === 'debit' ? (float) $tx->amount : 0,
                    'credit' => $tx->type === 'credit' ? (float) $tx->amount : 0,
                    'balance' => (float) $tx->balance_after,
                    'status' => 'completed',
                ];
            });

        return Inertia::render('SocietyAdmin/BankAccounts/Transactions', [
            'society' => $society->only(['id', 'name', 'code']),
            'bankAccount' => $bankAccount,
            'transactions' => $transactions,
            'filters' => $request->only(['start_date', 'end_date']),
        ]);
    }

    /**
     * Export the bank statement.
     */
    public function export(Request $request, Society $society, BankAccount $bankAccount): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $bankAccount->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $query = \App\Models\Transaction::where('bank_account_id', $bankAccount->id)
            // Exclude transactions linked to failed or pending payments
            ->whereNotExists(function ($query) {
                $query->select(\DB::raw(1))
                    ->from('payments')
                    ->whereColumn('payments.id', 'transactions.reference_id')
                    ->where('transactions.reference_type', 'App\\Models\\Payment')
                    ->whereIn('payments.status', ['failed', 'pending']);
            });

        // Filter by date
        if ($request->has('start_date') && $request->get('start_date')) {
            $query->whereDate('transaction_date', '>=', $request->get('start_date'));
        }

        if ($request->has('end_date') && $request->get('end_date')) {
            $query->whereDate('transaction_date', '<=', $request->get('end_date'));
        }

        // Determine opening and closing balance based on filters
        // This is complex because we just have running transactions.
        // For simplicity in this iteration, we will rely on the current balance and transaction history.
        // A more robust system would require querying previous transactions to calculate opening balance.

        $transactions = $query->with(['createdBy:id,name', 'reference'])
            ->orderBy('transaction_date') // Chronological order for statement
            ->orderBy('id')
            ->get()
            ->map(function ($tx) {
                return [
                    'id' => $tx->id,
                    'type' => $tx->type,
                    'date' => $tx->transaction_date->format('d-M-Y'),
                    'description' => $tx->description,
                    'reference' => $this->formatReference($tx->reference),
                    'payment_method' => $tx->payment_method ?? 'N/A',
                    'debit' => $tx->type === 'debit' ? (float) $tx->amount : 0,
                    'credit' => $tx->type === 'credit' ? (float) $tx->amount : 0,
                    'balance' => (float) $tx->balance_after,
                ];
            });

        // Calculate opening balance if filter is applied
        $openingBalance = 0;
        if ($transactions->isNotEmpty()) {
            $firstTx = $transactions->first();
            if ($firstTx['type'] === 'credit') {
                $openingBalance = $firstTx['balance'] - $firstTx['credit'];
            } else {
                $openingBalance = $firstTx['balance'] + $firstTx['debit'];
            }
        } elseif ($request->has('start_date')) {
            // Find the last transaction before the start date
             $lastTxBefore = \App\Models\Transaction::where('bank_account_id', $bankAccount->id)
                ->whereDate('transaction_date', '<', $request->get('start_date'))
                ->latest('transaction_date')
                ->latest('id')
                ->first();
            
            if ($lastTxBefore) {
                $openingBalance = (float) $lastTxBefore->balance_after;
            }
        }

        return Inertia::render('SocietyAdmin/BankAccounts/Statement', [
            'society' => $society,
            'bankAccount' => $bankAccount,
            'transactions' => $transactions,
            'filters' => $request->only(['start_date', 'end_date']),
            'openingBalance' => $openingBalance,
            'closingBalance' => $transactions->last()['balance'] ?? $openingBalance,
        ]);
    }

    /**
     * Reconcile the bank account.
     */
    public function reconcile(Society $society, BankAccount $bankAccount): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $bankAccount->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        $bankAccount->update([
            'last_reconciled_at' => now(),
            'updated_by' => $user->id,
        ]);

        return redirect()->route('society.bank-accounts.index', $society)
            ->with('success', 'Bank account reconciled successfully.');
    }

    private function formatReference($reference)
    {
        if (! $reference) {
            return '-';
        }

        if ($reference instanceof \App\Models\Payment) {
            return 'Payment #' . $reference->payment_no;
        }

        if ($reference instanceof \App\Models\Expense) {
            return 'Expense #' . $reference->expense_no;
        }

        if ($reference instanceof \App\Models\Bill) {
            return 'Bill #' . $reference->bill_no;
        }
        
        if ($reference instanceof \App\Models\FacilityBooking) {
            return 'Booking #' . $reference->id; 
        }

        if ($reference instanceof \App\Models\Advertisement) {
            return 'Ad: ' . $reference->title;
        }

        return class_basename($reference) . ' #' . $reference->id;
    }
}
