<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Http\Requests\SocietyAdmin\StoreAnnouncementRequest;
use App\Http\Requests\SocietyAdmin\UpdateAnnouncementRequest;
use App\Models\Announcement;
use App\Models\Building;
use App\Models\Society;
use App\Models\Unit;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class AnnouncementController extends Controller
{
    /**
     * Display a listing of announcements.
     */
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $query = Announcement::where('society_id', $society->id)
            ->with('createdBy:id,name');

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                    ->orWhere('content', 'like', "%{$search}%");
            });
        }

        // Filter by type
        if ($request->has('type') && $request->get('type') !== 'all') {
            $query->where('type', $request->get('type'));
        }

        // Filter by priority
        if ($request->has('priority') && $request->get('priority') !== 'all') {
            $query->where('priority', $request->get('priority'));
        }

        // Filter by published status
        if ($request->has('is_published') && $request->get('is_published') !== 'all') {
            $query->where('is_published', $request->get('is_published') === '1');
        }

        $announcements = $query->latest()->paginate(15)->withQueryString();

        // Calculate stats
        $stats = [
            'total' => Announcement::where('society_id', $society->id)->count(),
            'active' => Announcement::where('society_id', $society->id)
                ->where('is_published', true)
                ->where(function ($q) {
                    $q->whereNull('end_date')
                        ->orWhere('end_date', '>=', now());
                })->count(),
            'total_views' => Announcement::where('society_id', $society->id)->sum('views_count'),
            'acknowledgements' => Announcement::where('society_id', $society->id)
                ->where('require_acknowledgement', true)
                ->sum('acknowledged_count'),
        ];

        // Get buildings for targeting
        $buildings = Building::where('society_id', $society->id)
            ->orderBy('name')
            ->get(['id', 'name', 'building_no']);

        return Inertia::render('SocietyAdmin/Announcements/Index', [
            'society' => $society->only(['id', 'name', 'code']),
            'announcements' => $announcements,
            'filters' => $request->only(['search', 'type', 'priority', 'is_published']),
            'stats' => $stats,
            'buildings' => $buildings,
        ]);
    }

    /**
     * Show the form for creating a new announcement.
     */
    public function create(Society $society): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // Get units for targeting
        $units = Unit::where('society_id', $society->id)
            ->where('status', 'occupied')
            ->with('user:id,name')
            ->orderBy('unit_no')
            ->get(['id', 'unit_no', 'user_id']);

        return Inertia::render('SocietyAdmin/Announcements/Create', [
            'society' => $society->only(['id', 'name', 'code']),
            'units' => $units,
        ]);
    }

    /**
     * Store a newly created announcement.
     */
    public function store(StoreAnnouncementRequest $request, Society $society): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        $data = $request->validated();
        $data['society_id'] = $society->id;
        $data['created_by'] = $user->id;

        // Convert target arrays
        if ($request->has('target_roles') && is_array($request->get('target_roles'))) {
            $data['target_roles'] = array_filter($request->get('target_roles'));
        } else {
            $data['target_roles'] = null;
        }

        if ($request->has('target_buildings') && is_array($request->get('target_buildings'))) {
            $data['target_buildings'] = array_map('intval', array_filter($request->get('target_buildings')));
        } else {
            $data['target_buildings'] = null;
        }

        if ($request->has('target_units') && is_array($request->get('target_units'))) {
            $data['target_units'] = array_map('intval', array_filter($request->get('target_units')));
        } else {
            $data['target_units'] = null;
        }

        $announcement = Announcement::create($data);

        // Send notification if published immediately
        if ($announcement->is_published) {
            $this->sendAnnouncementNotification($announcement);
        }

        return redirect()->route('society.announcements.index', $society)
            ->with('success', 'Announcement created successfully.');
    }

    /**
     * Show the form for editing the specified announcement.
     */
    public function edit(Society $society, Announcement $announcement): Response
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $announcement->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No edit permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only create announcements.');
        }

        $announcement->load('createdBy:id,name');

        // Get units for targeting
        $units = Unit::where('society_id', $society->id)
            ->where('status', 'occupied')
            ->with('user:id,name')
            ->orderBy('unit_no')
            ->get(['id', 'unit_no', 'user_id']);

        return Inertia::render('SocietyAdmin/Announcements/Edit', [
            'society' => $society->only(['id', 'name', 'code']),
            'announcement' => $announcement,
            'units' => $units,
        ]);
    }

    /**
     * Update the specified announcement.
     */
    public function update(UpdateAnnouncementRequest $request, Society $society, Announcement $announcement): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $announcement->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No update permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only create announcements.');
        }

        $data = $request->validated();

        // Convert target arrays
        if ($request->has('target_roles') && is_array($request->get('target_roles'))) {
            $data['target_roles'] = array_filter($request->get('target_roles'));
        } else {
            $data['target_roles'] = null;
        }

        if ($request->has('target_buildings') && is_array($request->get('target_buildings'))) {
            $data['target_buildings'] = array_map('intval', array_filter($request->get('target_buildings')));
        } else {
            $data['target_buildings'] = null;
        }

        if ($request->has('target_units') && is_array($request->get('target_units'))) {
            $data['target_units'] = array_map('intval', array_filter($request->get('target_units')));
        } else {
            $data['target_units'] = null;
        }

        $announcement->update($data);

        return redirect()->route('society.announcements.index', $society)
            ->with('success', 'Announcement updated successfully.');
    }

    /**
     * Remove the specified announcement.
     */
    public function destroy(Society $society, Announcement $announcement): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $announcement->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No delete permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin can only create announcements.');
        }

        $announcement->delete();

        return redirect()->route('society.announcements.index', $society)
            ->with('success', 'Announcement deleted successfully.');
    }

    /**
     * Toggle publish status of the announcement.
     */
    public function togglePublish(Society $society, Announcement $announcement): RedirectResponse
    {
        $user = Auth::user();

        if ($user->society_id !== $society->id || $announcement->society_id !== $society->id) {
            abort(403, 'Unauthorized access.');
        }

        // Sub-Admin: No publish/unpublish permission
        if ($user->hasRole('sub_admin')) {
            abort(403, 'Sub-Admin cannot publish/unpublish announcements.');
        }

        $announcement->update([
            'is_published' => ! $announcement->is_published,
        ]);

        // Send notification if published
        if ($announcement->is_published) {
            $this->sendAnnouncementNotification($announcement);
        }

        $message = $announcement->is_published
            ? 'Announcement published successfully.'
            : 'Announcement unpublished successfully.';

        return redirect()->back()
            ->with('success', $message);
    }

    /**
     * Helper to retrieve target users and send notification.
     */
    protected function sendAnnouncementNotification(Announcement $announcement): void
    {
        try {
            $query = \App\Models\User::where('society_id', $announcement->society_id)
                ->whereNotNull('fcm_token');

            // 1. Target by Units
            if (! empty($announcement->target_units)) {
                $query->whereIn('unit_id', $announcement->target_units);
            }
            // 2. Target by Buildings (if unit filter not present/exclusive, usually OR or drilldown)
            // If target_units is empty but target_buildings is not
            elseif (! empty($announcement->target_buildings)) {
                $query->whereHas('unit', function ($q) use ($announcement) {
                    $q->whereIn('building_id', $announcement->target_buildings);
                });
            }

            // 3. Target by Roles (e.g. resident, guard). This acts as a filter on the result.
            // If specific roles are targeted, we filter efficiently using whereHas roles
            if (! empty($announcement->target_roles)) {
                $query->whereHas('role', function ($q) use ($announcement) {
                    $q->whereIn('name', $announcement->target_roles);
                });
            }

            // If no filters (null targets), it sends to everyone in society (which is default $query)

            $users = $query->get();

            if ($users->isEmpty()) {
                return;
            }

            $firebaseService = app(\App\Services\FirebaseNotificationService::class);
            $firebaseService->sendToUsers(
                $users->all(),
                'New Announcement: ' . $announcement->title,
                \Illuminate\Support\Str::limit(strip_tags($announcement->content), 100),
                [
                    'type' => 'announcement',
                    'id' => (string) $announcement->id,
                    'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                ]
            );

        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to send announcement notification: ' . $e->getMessage());
        }
    }
}
