<?php

namespace App\Http\Controllers\SocietyAdmin;

use App\Http\Controllers\Controller;
use App\Models\Advertisement;
use App\Services\FirebaseNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

class AdminAdController extends Controller
{
    /**
     * Pending Ad Requests
     */
    public function requests(Request $request)
    {
        $ads = Advertisement::where('society_id', Auth::user()->society_id)
            ->where('status', 'pending_approval')
            ->with(['user.unit.building', 'slot'])
            ->orderBy('created_at', 'asc');

        if ($request->search) {
            $search = $request->search;
            $ads->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhereHas('user', function($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        return Inertia::render('SocietyAdmin/Monetization/Requests', [
            'ads' => $ads->paginate(10)->withQueryString(),
            'filters' => $request->only(['search']),
            'society' => Auth::user()->society,
        ]);
    }

    /**
     * Active/History Ads
     */
    public function index(Request $request)
    {
        $query = Advertisement::where('society_id', Auth::user()->society_id)
            ->where('status', '!=', 'pending_approval');

        // Filters
        if ($request->status) {
            $query->where('status', $request->status);
        }
        
        // Example: Only Active by default if not specified? Or All?
        // User asked for filters: Active, Expired, All.

        if ($request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhereHas('user', function($q) use ($search) {
                      $q->where('name', 'like', "%{$search}%");
                  });
            });
        }

        $ads = $query->with(['slot', 'user'])
            ->orderBy('created_at', 'desc')
            ->paginate(10)
            ->withQueryString();

        return Inertia::render('SocietyAdmin/Monetization/Index', [
            'ads' => $ads,
            'filters' => $request->only(['status', 'search']),
            'society' => Auth::user()->society,
        ]);
    }

    public function approve(Request $request, Advertisement $advertisement, FirebaseNotificationService $firebaseService)
    {
        if ($advertisement->society_id !== Auth::user()->society_id) abort(403);

        $advertisement->update(['status' => 'active']); // Or 'approved' if start date is future? 
        // User spec: "Update status to 'active' (or 'approved')."
        
        // Create Payment Record for Revenue Tracking
        if ($advertisement->price > 0) {
            // Determine primary bank account
            $primaryAccount = \App\Models\BankAccount::where('society_id', $advertisement->society_id)
                ->where('is_primary', true)
                ->first();

            // Fallback to first available account
            if (!$primaryAccount) {
                $primaryAccount = \App\Models\BankAccount::where('society_id', $advertisement->society_id)->first();
            }

            $payment = \App\Models\Payment::create([
                'society_id' => $advertisement->society_id,
                'user_id' => $advertisement->user_id,
                'bill_id' => null, // Not linked to a bill
                'payment_no' => 'PAY-AD-' . strtoupper(\Illuminate\Support\Str::random(8)),
                'amount' => $advertisement->price,
                'payment_method' => 'online', 
                'transaction_id' => $advertisement->payment_reference,
                'bank_account_id' => $primaryAccount ? $primaryAccount->id : null,
                'status' => 'completed',
                'payment_date' => now(),
                'notes' => 'Advertisement Revenue: ' . $advertisement->title,
                'collected_by' => Auth::id(), // System or Admin who approved it? Admin approved it.
                'collector_role' => 'admin',
            ]);

            // Record Transaction
            if ($primaryAccount) {
                app(\App\Services\TransactionService::class)->recordIncome(
                    $payment->amount,
                    $primaryAccount->id,
                    'ad_revenue',
                    $advertisement, // Reference the Ad, or the Payment? Plan says "Ad Revenue". Can ref Payment too.
                    // Implementation Plan said "Transaction... Reference Polymorphic".
                    // Usually we reference the Source (Ad). But Payment also exists.
                    // Let's reference the Payment for consistency with other payments, OR reference Ad to show "Ad Revenue".
                    // The Payment model morphs to Transaction usually. 
                    // Let's reference the PAYMENT as the source of money movement, but category is 'ad_revenue'.
                    $payment, 
                    'online',
                    'Advertisement Revenue: ' . $advertisement->title,
                    Auth::id()
                );
            }
        }

        // Notify
        $firebaseService->sendToUser($advertisement->user, "Ad Approved", "Your ad '{$advertisement->title}' has been approved!");

        return redirect()->back()->with('success', 'Ad approved successfully.');
    }

    public function reject(Request $request, Advertisement $advertisement, FirebaseNotificationService $firebaseService)
    {
        if ($advertisement->society_id !== Auth::user()->society_id) abort(403);
        
        $request->validate(['rejection_reason' => 'required|string']);

        $advertisement->update([
            'status' => 'rejected',
            'admin_notes' => $request->rejection_reason,
        ]);

        // Notify
        $firebaseService->sendToUser($advertisement->user, "Ad Rejected", "Reason: {$request->rejection_reason}");

        return redirect()->back()->with('success', 'Ad rejected.');
    }

    public function destroy(Advertisement $advertisement)
    {
        if ($advertisement->society_id !== Auth::user()->society_id) abort(403);
        
        $advertisement->delete();

        return redirect()->back()->with('success', 'Ad deleted.');
    }
}
