<?php

namespace App\Http\Controllers\Resident;

use App\Http\Controllers\Controller;
use App\Models\Facility;
use App\Models\FacilityBooking;
use App\Models\FacilitySlot;
use App\Services\PaymentGateway\SocietyPaymentGatewayFactory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class FacilityBookingController extends Controller
{
    /**
     * Create payment order before booking (supports all 4 gateways)
     */
    /**
     * Create payment order for facility booking
     * Supports multiple payment gateways: Razorpay, Stripe, PayU, Cashfree
     */
    public function createOrder(Request $request)
    {
        $validated = $request->validate([
            'facility_id' => 'required|exists:facilities,id',
            'slot_id' => 'required|exists:facility_slots,id',
            'booking_date' => 'required|date|after:today',
            'purpose' => 'nullable|string|max:255',
            'guest_count' => 'nullable|integer|min:1',
            'gateway' => 'nullable|string|in:razorpay,stripe,payu,cashfree',
        ]);

        // Check if facility and slot are active
        $facility = Facility::where('id', $validated['facility_id'])
            ->where('is_active', true)
            ->with('society.settings')
            ->firstOrFail();

        $societySettings = $facility->society->settings;
        
        if (!$societySettings) {
            return response()->json([
                'success' => false,
                'message' => 'Society settings not configured'
            ], 400);
        }

        // Get gateway using factory
        $gatewayId = $validated['gateway'] ?? null;
        $gateway = $gatewayId 
            ? SocietyPaymentGatewayFactory::getGateway($societySettings, $gatewayId)
            : SocietyPaymentGatewayFactory::getDefaultGateway($societySettings);

        if (!$gateway) {
            return response()->json([
                'success' => false,
                'message' => 'No payment gateway configured for this society'
            ], 400);
        }

        $slot = FacilitySlot::where('id', $validated['slot_id'])
            ->where('facility_id', $validated['facility_id'])
            ->where('is_active', true)
            ->firstOrFail();

        // Check slot availability for the date
        $bookingsCount = FacilityBooking::where('facility_id', $validated['facility_id'])
            ->where('slot_id', $validated['slot_id'])
            ->where('booking_date', $validated['booking_date'])
            ->whereIn('booking_status', ['pending', 'confirmed'])
            ->count();

        if ($bookingsCount >= $slot->max_bookings_per_slot) {
            return response()->json([
                'success' => false,
                'message' => 'Selected slot is not available for the chosen date'
            ], 409);
        }

        try {
            // Create order using gateway factory
            $orderData = $gateway->createOrder($slot->price, 'INR', [
                'receipt' => 'facility_booking_' . time() . '_' . auth()->id(),
                'notes' => [
                    'facility_id' => $validated['facility_id'],
                    'slot_id' => $validated['slot_id'],
                    'booking_date' => $validated['booking_date'],
                    'user_id' => auth()->id(),
                    'society_id' => $facility->society_id,
                    'purpose' => $validated['purpose'] ?? '',
                    'guest_count' => $validated['guest_count'] ?? 0,
                    'type' => 'facility_booking',
                ],
                'customer_name' => auth()->user()->name,
                'customer_email' => auth()->user()->email,
                'customer_phone' => auth()->user()->phone ?? '9999999999',
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Order created successfully',
                'data' => [
                    'gateway' => $gateway->getIdentifier(),
                    'order_id' => $orderData['order_id'],
                    'amount' => $slot->price * 100,
                    'currency' => 'INR',
                    'key' => $orderData['key'] ?? null,
                    'client_secret' => $orderData['client_secret'] ?? null,
                    'hash' => $orderData['hash'] ?? null,
                    'payment_session_id' => $orderData['payment_session_id'] ?? null,
                    'action_url' => $orderData['action_url'] ?? null,
                    'booking_data' => [
                        'facility_id' => $validated['facility_id'],
                        'slot_id' => $validated['slot_id'],
                        'booking_date' => $validated['booking_date'],
                        'purpose' => $validated['purpose'] ?? '',
                        'guest_count' => $validated['guest_count'] ?? 0,
                        'facility_name' => $facility->name,
                        'slot_name' => $slot->name,
                        'slot_time' => $slot->time_range ?? "{$slot->start_time} - {$slot->end_time}",
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            // Log error for debugging payment gateway issues
            Log::error('Facility booking order creation failed', [
                'error' => $e->getMessage(),
                'gateway' => $gateway->getIdentifier(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error creating order: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Verify payment and create booking (supports all 4 gateways)
     */
    /**
     * Verify payment and create facility booking
     * Handles payment verification for all supported gateways
     */
    public function verifyPayment(Request $request)
    {
        $validated = $request->validate([
            'gateway' => 'required|string|in:razorpay,stripe,payu,cashfree',
            'facility_id' => 'required|exists:facilities,id',
            'slot_id' => 'required|exists:facility_slots,id',
            'booking_date' => 'required|date|after:today',
            'purpose' => 'nullable|string|max:255',
            'guest_count' => 'nullable|integer|min:1',
            // Razorpay specific
            'razorpay_order_id' => 'required_if:gateway,razorpay|nullable|string',
            'razorpay_payment_id' => 'required_if:gateway,razorpay|nullable|string',
            'razorpay_signature' => 'required_if:gateway,razorpay|nullable|string',
            // Stripe specific
            'payment_intent_id' => 'required_if:gateway,stripe|nullable|string',
            // PayU specific
            'payu_txnid' => 'required_if:gateway,payu|nullable|string',
            'payu_mihpayid' => 'required_if:gateway,payu|nullable|string',
            'payu_status' => 'required_if:gateway,payu|nullable|string',
            'payu_hash' => 'required_if:gateway,payu|nullable|string',
            // Cashfree specific
            'cashfree_order_id' => 'required_if:gateway,cashfree|nullable|string',
        ]);

        try {
            // Get facility with society settings
            $facility = Facility::where('id', $validated['facility_id'])
                ->with('society.settings')
                ->firstOrFail();

            $societySettings = $facility->society->settings;

            if (!$societySettings) {
                return response()->json([
                    'success' => false,
                    'message' => 'Society settings not configured'
                ], 400);
            }

            // Get gateway
            $gateway = SocietyPaymentGatewayFactory::getGateway($societySettings, $validated['gateway']);

            if (!$gateway) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment gateway not configured'
                ], 400);
            }

            // Verify payment using gateway
            $result = $gateway->verifyPayment($request->all());

            if (!$result['success']) {
                // Log payment verification failure for debugging
                Log::warning('Facility booking payment verification failed', [
                    'gateway' => $validated['gateway'],
                    'error' => $result['error'] ?? 'Unknown error',
                ]);
                return response()->json([
                    'success' => false,
                    'message' => $result['error'] ?? 'Payment verification failed'
                ], 400);
            }

            // Double check slot availability
            $slot = FacilitySlot::findOrFail($validated['slot_id']);
            $bookingsCount = FacilityBooking::where('facility_id', $validated['facility_id'])
                ->where('slot_id', $validated['slot_id'])
                ->where('booking_date', $validated['booking_date'])
                ->whereIn('booking_status', ['pending', 'confirmed'])
                ->count();

            if ($bookingsCount >= $slot->max_bookings_per_slot) {
                return response()->json([
                    'success' => false,
                    'message' => 'Slot is no longer available. Please contact support for refund.'
                ], 409);
            }

            // Create Booking
            $booking = FacilityBooking::create([
                'facility_id' => $validated['facility_id'],
                'slot_id' => $validated['slot_id'],
                'user_id' => auth()->id(),
                'booked_by' => auth()->id(),
                'society_id' => $facility->society_id,
                'booking_date' => $validated['booking_date'],
                'amount' => $slot->price,
                'payment_status' => 'paid',
                'payment_method' => 'online',
                'payment_gateway' => $validated['gateway'],
                'booking_status' => $facility->requires_approval ? 'pending' : 'confirmed',
                'purpose' => $validated['purpose'],
                'guest_count' => $validated['guest_count'],
                'transaction_id' => $result['payment_id'] ?? null,
                'payment_response' => json_encode($request->all()),
            ]);

            // Load related models for response
            $booking->load(['facility', 'slot', 'user']);

            return response()->json([
                'success' => true,
                'message' => 'Payment verified and booking created successfully',
                'data' => $booking
            ]);

        } catch (\Exception $e) {
            // Log error for debugging payment verification and booking creation issues
            Log::error('Facility booking payment verification error', [
                'error' => $e->getMessage(),
                'gateway' => $validated['gateway'] ?? null,
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Error creating booking: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Log payment failure
     */
    public function paymentFailed(Request $request)
    {
        $validated = $request->validate([
            'gateway' => 'nullable|string',
            'order_id' => 'nullable|string',
            'error_code' => 'nullable|string',
            'error_description' => 'nullable|string',
            'facility_id' => 'required|exists:facilities,id',
            'slot_id' => 'required|exists:facility_slots,id'
        ]);

        // Log payment failure for monitoring and debugging
        Log::warning('Facility booking payment failed', [
            'user_id' => auth()->id(),
            'gateway' => $validated['gateway'],
            'order_id' => $validated['order_id'],
            'error_code' => $validated['error_code'],
            'error_description' => $validated['error_description'],
            'facility_id' => $validated['facility_id'],
            'slot_id' => $validated['slot_id'],
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Payment failure logged'
        ]);
    }
}
