<?php

namespace App\Http\Controllers;

use App\Models\Society;
use App\Models\SocietySetting;
use App\Models\Subscription;
use App\Models\SubscriptionPayment;
use App\Models\SubscriptionPlan;
use App\Models\SystemSetting;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Razorpay\Api\Api;
use Razorpay\Api\Errors\SignatureVerificationError;

class OnboardingController extends Controller
{
    /**
     * Store a newly created society and link it to the current user.
     * Creates subscription with 'pending_payment' status.
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255'],
            'phone' => ['required', 'string', 'max:20'],
            'address' => ['required', 'string', 'max:500'],
            'city' => ['required', 'string', 'max:100'],
            'state' => ['required', 'string', 'max:100'],
            'pincode' => ['required', 'string', 'max:20'],
            'country' => ['required', 'string', 'max:100'],
            'plan_id' => ['required', 'exists:subscription_plans,id'],
            'billing_cycle' => ['required', 'in:monthly,quarterly,annual'],
            'razorpay_payment_id' => ['required', 'string'],
            'razorpay_order_id' => ['required', 'string'],
            'razorpay_signature' => ['required', 'string'],
        ]);

        try {
            // Verify Razorpay payment first
            $razorpayKey = SystemSetting::getValue('razorpay_key');
            $razorpaySecret = SystemSetting::getValue('razorpay_secret');

            if (!$razorpayKey || !$razorpaySecret) {
                return back()->withErrors(['error' => 'Payment gateway not configured. Please contact support.'])->withInput();
            }

            $api = new Api($razorpayKey, $razorpaySecret);

            // Verify signature
            $attributes = [
                'razorpay_order_id' => $request->razorpay_order_id,
                'razorpay_payment_id' => $request->razorpay_payment_id,
                'razorpay_signature' => $request->razorpay_signature,
            ];

            $api->utility->verifyPaymentSignature($attributes);

            return DB::transaction(function () use ($request, $api) {
                $user = Auth::user();

                // Create Society
                $society = Society::create([
                    'name' => $request->name,
                    'email' => $request->email,
                    'phone' => $request->phone,
                    'address' => $request->address,
                    'city' => $request->city,
                    'state' => $request->state,
                    'pincode' => $request->pincode,
                    'registration_no' => 'SOC-' . strtoupper(uniqid()),
                    'status' => 'active',
                ]);

                // Link User to Society
                $user->update(['society_id' => $society->id]);

                // Create Subscription
                $plan = SubscriptionPlan::findOrFail($request->plan_id);
                $startDate = now();
                $billingCycle = $request->billing_cycle;

                $endDate = match ($billingCycle) {
                    'monthly' => $startDate->copy()->addMonth(),
                    'quarterly' => $startDate->copy()->addMonths(3),
                    'annual' => $startDate->copy()->addYear(),
                    default => $startDate->copy()->addMonth(),
                };

                $nextBillingDate = $endDate->copy();

                // Calculate actual amount paid
                $amount = match ($billingCycle) {
                    'monthly' => $plan->monthly_price,
                    'quarterly' => $plan->quarterly_price ?? ($plan->monthly_price * 3),
                    'annual' => $plan->annual_price,
                    default => $plan->monthly_price,
                };

                $subscription = Subscription::create([
                    'society_id' => $society->id,
                    'plan_id' => $plan->id,
                    'billing_cycle' => $billingCycle,
                    'status' => 'active', // Payment verified, subscription is active
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                    'next_billing_date' => $nextBillingDate,
                    'auto_renew' => true,
                    'payment_gateway' => 'razorpay',
                    'plan_name' => $plan->name,
                    'plan_display_name' => $plan->display_name,
                    'monthly_price' => $plan->monthly_price,
                    'annual_price' => $plan->annual_price,
                    'max_users' => $plan->max_users,
                    'max_units' => $plan->max_units,
                    'max_buildings' => $plan->max_buildings,
                    'storage_gb' => $plan->storage_gb,
                    'sms_enabled' => $plan->sms_enabled,
                    'whatsapp_enabled' => $plan->whatsapp_enabled,
                    'email_enabled' => $plan->email_enabled,
                    'fcm_enabled' => $plan->fcm_enabled,
                    'advanced_reports' => $plan->advanced_reports,
                    'api_access' => $plan->api_access,
                    'trial_start_date' => null,
                    'trial_end_date' => null,
                ]);

                // Record the payment
                SubscriptionPayment::create([
                    'subscription_id' => $subscription->id,
                    'society_id' => $society->id,
                    'razorpay_order_id' => $request->razorpay_order_id,
                    'razorpay_payment_id' => $request->razorpay_payment_id,
                    'razorpay_signature' => $request->razorpay_signature,
                    'amount' => $amount,
                    'currency' => 'INR',
                    'status' => 'completed',
                    'payment_gateway' => 'razorpay',
                    'paid_at' => now(),
                    'notes' => 'Initial subscription payment',
                ]);

                // Create Default Settings
                SocietySetting::create([
                    'society_id' => $society->id,
                    'currency' => 'INR',
                    'timezone' => 'Asia/Kolkata',
                    'date_format' => 'd/m/Y',
                    'time_format' => 'H:i',
                    'language' => 'en',
                ]);

                // Society onboarding completed successfully
                // No logging needed for successful operations

                // Send welcome and activation emails
                \App\Services\EmailService::sendWelcomeEmail($society, $user);
                \App\Services\EmailService::sendSubscriptionActivatedEmail($society, $subscription);

                return redirect()->route('dashboard')->with('success', 'Society created successfully!');
            });
        } catch (SignatureVerificationError $e) {
            // Log payment signature verification failure (security-critical)
            Log::error('Razorpay signature verification failed', [
                'error' => $e->getMessage(),
                'order_id' => $request->razorpay_order_id,
            ]);
            return back()->withErrors(['error' => 'Payment verification failed. Please try again.'])->withInput();
        } catch (\Exception $e) {
            // Log error for debugging onboarding failures
            Log::error('Society onboarding error', ['error' => $e->getMessage()]);
            return back()->withErrors(['error' => 'An error occurred while creating the society: ' . $e->getMessage()])->withInput();
        }
    }

    /**
     * Create a Razorpay order for subscription payment.
     */
    public function createPaymentOrder(Request $request): JsonResponse
    {
        $request->validate([
            'plan_id' => ['required', 'exists:subscription_plans,id'],
            'billing_cycle' => ['required', 'in:monthly,quarterly,annual'],
        ]);

        try {
            $razorpayKey = SystemSetting::getValue('razorpay_key');
            $razorpaySecret = SystemSetting::getValue('razorpay_secret');

            if (!$razorpayKey || !$razorpaySecret) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment gateway not configured. Please contact support.',
                ], 400);
            }

            $plan = SubscriptionPlan::findOrFail($request->plan_id);
            $billingCycle = $request->billing_cycle;

            // Calculate amount based on billing cycle
            $amount = match ($billingCycle) {
                'monthly' => $plan->monthly_price,
                'quarterly' => $plan->quarterly_price ?? ($plan->monthly_price * 3),
                'annual' => $plan->annual_price,
                default => $plan->monthly_price,
            };

            $api = new Api($razorpayKey, $razorpaySecret);

            $order = $api->order->create([
                'amount' => $amount * 100, // Razorpay amount in paise
                'currency' => 'INR',
                'receipt' => 'sub_' . uniqid(),
                'notes' => [
                    'plan_id' => $plan->id,
                    'plan_name' => $plan->name,
                    'billing_cycle' => $billingCycle,
                    'user_id' => Auth::id(),
                ],
            ]);

            return response()->json([
                'success' => true,
                'order_id' => $order->id,
                'amount' => $amount,
                'currency' => 'INR',
                'key' => $razorpayKey,
                'plan' => [
                    'id' => $plan->id,
                    'name' => $plan->display_name,
                    'billing_cycle' => $billingCycle,
                ],
            ]);
        } catch (\Exception $e) {
            // Log error for debugging payment gateway issues
            Log::error('Failed to create Razorpay order', [
                'error' => $e->getMessage(),
                'plan_id' => $request->plan_id,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to create payment order: ' . $e->getMessage(),
            ], 500);
        }
    }
}
