<?php

namespace App\Http\Controllers;

use App\Models\SuperAdmin;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Inertia\Inertia;

class InstallController extends Controller
{
    /**
     * Constructor - Switch to file sessions during installation
     */
    public function __construct()
    {
        // Use file sessions during installation (before migrations run)
        if (config('session.driver') === 'database') {
            config(['session.driver' => 'file']);
        }
    }

    /**
     * Required PHP extensions for the application
     */
    protected array $requiredExtensions = [
        'pdo',
        'mbstring',
        'openssl',
        'tokenizer',
        'json',
        'curl',
        'xml',
        'fileinfo',
    ];

    /**
     * Directories that need to be writable
     */
    protected array $writableDirectories = [
        'storage',
        'storage/app',
        'storage/framework',
        'storage/logs',
        'bootstrap/cache',
    ];

    /**
     * Redirect to the current installation step
     */
    public function index()
    {
        if ($this->isInstalled()) {
            return redirect()->route('super-admin.login');
        }

        $state = $this->getInstallState();

        return match ($state['current_step']) {
            1 => redirect()->route('install.welcome'),
            2 => redirect()->route('install.requirements'),
            3 => redirect()->route('install.database'),
            4 => redirect()->route('install.admin'),
            5 => redirect()->route('install.finish'),
            default => redirect()->route('install.welcome'),
        };
    }

    /**
     * Step 1: Welcome page
     */
    public function welcome()
    {
        if ($this->isInstalled()) {
            return redirect()->route('super-admin.login');
        }

        $this->updateInstallState(['current_step' => 1]);

        return Inertia::render('Install/Welcome');
    }

    /**
     * Step 2: Requirements check page
     */
    public function requirements()
    {
        if ($this->isInstalled()) {
            return redirect()->route('super-admin.login');
        }

        $requirements = $this->checkSystemRequirements();
        $this->updateInstallState(['current_step' => 2]);

        return Inertia::render('Install/Requirements', [
            'requirements' => $requirements,
            'allPassed' => $this->allRequirementsPassed($requirements),
        ]);
    }

    /**
     * POST: Verify requirements and proceed
     */
    public function checkRequirements(Request $request)
    {
        $requirements = $this->checkSystemRequirements();

        if (!$this->allRequirementsPassed($requirements)) {
            return back()->withErrors(['requirements' => 'Please fix all requirements before proceeding.']);
        }

        $this->updateInstallState(['requirements_passed' => true, 'current_step' => 3]);

        return redirect()->route('install.database');
    }

    /**
     * Step 3: Database configuration page
     */
    public function database()
    {
        if ($this->isInstalled()) {
            return redirect()->route('super-admin.login');
        }

        $state = $this->getInstallState();
        if (!$state['requirements_passed']) {
            return redirect()->route('install.requirements');
        }

        // Temporarily switch to file sessions during installation (before migrations run)
        if (config('session.driver') === 'database') {
            config(['session.driver' => 'file']);
        }

        $this->updateInstallState(['current_step' => 3]);

        return Inertia::render('Install/Database', [
            'currentConnection' => config('database.default'),
        ]);
    }

    /**
     * POST: Configure database and run migrations
     */
    public function configureDatabase(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'db_connection' => 'required|in:sqlite,mysql',
            'db_host' => 'required_if:db_connection,mysql',
            'db_port' => 'required_if:db_connection,mysql',
            'db_database' => 'required_if:db_connection,mysql',
            'db_username' => 'required_if:db_connection,mysql',
            'db_password' => 'nullable',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator);
        }

        $dbConnection = $request->input('db_connection');

        try {
            // Update environment file
            $this->updateEnvFile($dbConnection, $request->all(), $request);

            // Clear all caches before proceeding
            Artisan::call('config:clear');
            Artisan::call('cache:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');

            // Ensure cache driver is file during installation
            config(['cache.default' => 'file']);

            // Apply database config at runtime for this request.
            // Updating `.env` does NOT change the current PHP process env, so
            // we must set config() + reconnect before running migrations.
            $this->applyDatabaseRuntimeConfig($dbConnection, $request->all());

            // Test database connection
            if ($dbConnection === 'mysql') {
                $this->testMySQLConnection($request->all());
            } else {
                $this->createSQLiteDatabase();
            }

            // Re-apply config after creating SQLite DB file (it deletes/recreates the file)
            $this->applyDatabaseRuntimeConfig($dbConnection, $request->all());

            // Run migrations
            $migrationOutput = Artisan::call('migrate', ['--force' => true]);
            
            // Verify migrations ran by checking if tables exist (especially for SQLite)
            if ($dbConnection === 'sqlite') {
                $dbPath = database_path('database.sqlite');
                if (!file_exists($dbPath) || filesize($dbPath) === 0) {
                    throw new \Exception('SQLite database file is empty after migrations. Please check file permissions.');
                }
                
                try {
                    $pdo = new \PDO('sqlite:' . $dbPath);
                    $pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
                    $tables = $pdo->query("SELECT name FROM sqlite_master WHERE type='table'")->fetchAll(\PDO::FETCH_COLUMN);
                    
                    if (empty($tables)) {
                        throw new \Exception('Migrations completed but no tables were created. Please check migration files.');
                    }
                } catch (\PDOException $e) {
                    throw new \Exception('Cannot verify database tables: ' . $e->getMessage());
                }
            }

            // Run essential seeders
            $this->runEssentialSeeders();

            // Create storage symlink for uploaded files (logo, favicon, etc.)
            Artisan::call('storage:link');

            $this->updateInstallState(['database_configured' => true, 'current_step' => 4]);

            return redirect()->route('install.admin');
        } catch (\Exception $e) {
            return back()->withErrors(['database' => 'Database configuration failed: ' . $e->getMessage()]);
        }
    }

    /**
     * Step 4: Super admin creation page
     */
    public function admin()
    {
        if ($this->isInstalled()) {
            return redirect()->route('super-admin.login');
        }

        $state = $this->getInstallState();
        if (!$state['database_configured']) {
            return redirect()->route('install.database');
        }

        $this->updateInstallState(['current_step' => 4]);

        return Inertia::render('Install/Admin');
    }

    /**
     * POST: Create super admin account
     */
    public function createAdmin(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:super_admins,email',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator);
        }

        try {
            SuperAdmin::create([
                'name' => $request->input('name'),
                'email' => $request->input('email'),
                'password' => Hash::make($request->input('password')),
                'email_verified_at' => now(),
            ]);

            $this->updateInstallState([
                'admin_created' => true,
                'admin_email' => $request->input('email'),
                'current_step' => 5,
            ]);

            return redirect()->route('install.finish');
        } catch (\Exception $e) {
            return back()->withErrors(['admin' => 'Failed to create admin: ' . $e->getMessage()]);
        }
    }

    /**
     * Step 5: Installation complete page
     */
    public function finish()
    {
        if ($this->isInstalled()) {
            return redirect()->route('super-admin.login');
        }

        $state = $this->getInstallState();
        if (!$state['admin_created']) {
            return redirect()->route('install.admin');
        }

        $loginUrl = route('super-admin.login');

        return Inertia::render('Install/Finish', [
            'loginUrl' => $loginUrl,
            'adminEmail' => $state['admin_email'] ?? '',
        ]);
    }

    /**
     * POST: Mark installation as complete
     */
    public function complete(Request $request)
    {
        $state = $this->getInstallState();

        if (!$state['admin_created']) {
            return redirect()->route('install.admin');
        }

        // Create installed.json file
        Storage::put('installed.json', json_encode([
            'installed_at' => now()->toIso8601String(),
            'version' => config('app.version', '1.0.0'),
            'admin_email' => $state['admin_email'] ?? '',
        ]));

        // Remove install state file
        Storage::delete('install_state.json');

        return redirect()->route('super-admin.login');
    }

    /**
     * Check if application is already installed
     */
    protected function isInstalled(): bool
    {
        return Storage::exists('installed.json');
    }

    /**
     * Get current installation state
     */
    protected function getInstallState(): array
    {
        if (Storage::exists('install_state.json')) {
            return json_decode(Storage::get('install_state.json'), true);
        }

        return [
            'current_step' => 1,
            'requirements_passed' => false,
            'database_configured' => false,
            'admin_created' => false,
            'admin_email' => null,
        ];
    }

    /**
     * Update installation state
     */
    protected function updateInstallState(array $data): void
    {
        $state = $this->getInstallState();
        $state = array_merge($state, $data);
        Storage::put('install_state.json', json_encode($state));
    }

    /**
     * Check all system requirements
     */
    protected function checkSystemRequirements(): array
    {
        $requirements = [
            'php' => [
                'name' => 'PHP Version',
                'required' => '8.2.0',
                'current' => PHP_VERSION,
                'passed' => version_compare(PHP_VERSION, '8.2.0', '>='),
            ],
            'extensions' => [],
            'directories' => [],
        ];

        // Check PHP extensions
        foreach ($this->requiredExtensions as $ext) {
            $requirements['extensions'][$ext] = [
                'name' => $ext,
                'passed' => extension_loaded($ext),
            ];
        }

        // Check writable directories
        foreach ($this->writableDirectories as $dir) {
            $path = base_path($dir);
            $requirements['directories'][$dir] = [
                'name' => $dir,
                'passed' => is_dir($path) && is_writable($path),
            ];
        }

        return $requirements;
    }

    /**
     * Check if all requirements passed
     */
    protected function allRequirementsPassed(array $requirements): bool
    {
        if (!$requirements['php']['passed']) {
            return false;
        }

        foreach ($requirements['extensions'] as $ext) {
            if (!$ext['passed']) {
                return false;
            }
        }

        foreach ($requirements['directories'] as $dir) {
            if (!$dir['passed']) {
                return false;
            }
        }

        return true;
    }

    /**
     * Update .env file with database configuration + app metadata.
     *
     * NOTE: `.env` updates do not automatically affect the current PHP process.
     */
    protected function updateEnvFile(string $connection, array $config, ?Request $request): void
    {
        // Stash the install URL into config array so we don't widen regex changes below.
        if ($request) {
            $config['__install_app_url'] = $request->root();
        }

        $envPath = base_path('.env');

        // Ensure .env exists during installation
        if (!file_exists($envPath) && file_exists(base_path('.env.example'))) {
            copy(base_path('.env.example'), $envPath);
        }

        $originalEnvContent = file_exists($envPath) ? file_get_contents($envPath) : '';
        $envContent = $originalEnvContent;

        if ($connection === 'sqlite') {
            $envContent = preg_replace('/^DB_CONNECTION=.*/m', 'DB_CONNECTION=sqlite', $envContent);
            // Comment out MySQL settings
            $envContent = preg_replace('/^DB_HOST=(.*)$/m', '# DB_HOST=$1', $envContent);
            $envContent = preg_replace('/^DB_PORT=(.*)$/m', '# DB_PORT=$1', $envContent);
            $envContent = preg_replace('/^DB_DATABASE=(?!.*sqlite)(.*)$/m', '# DB_DATABASE=$1', $envContent);
            $envContent = preg_replace('/^DB_USERNAME=(.*)$/m', '# DB_USERNAME=$1', $envContent);
            $envContent = preg_replace('/^DB_PASSWORD=(.*)$/m', '# DB_PASSWORD=$1', $envContent);
        } else {
            $envContent = preg_replace('/^DB_CONNECTION=.*/m', 'DB_CONNECTION=mysql', $envContent);
            $envContent = preg_replace('/^#?\s*DB_HOST=.*/m', 'DB_HOST=' . $config['db_host'], $envContent);
            $envContent = preg_replace('/^#?\s*DB_PORT=.*/m', 'DB_PORT=' . $config['db_port'], $envContent);
            $envContent = preg_replace('/^#?\s*DB_DATABASE=.*/m', 'DB_DATABASE=' . $config['db_database'], $envContent);
            $envContent = preg_replace('/^#?\s*DB_USERNAME=.*/m', 'DB_USERNAME=' . $config['db_username'], $envContent);
            $envContent = preg_replace('/^#?\s*DB_PASSWORD=.*/m', 'DB_PASSWORD=' . ($config['db_password'] ?? ''), $envContent);
        }

        // App metadata based on the install URL (domain)
        $baseUrl = '';
        if (isset($config['__install_app_url']) && is_string($config['__install_app_url']) && $config['__install_app_url'] !== '') {
            $baseUrl = rtrim($config['__install_app_url'], '/');
        } else {
            $baseUrl = rtrim((string) config('app.url', ''), '/');
        }

        $host = (string) (parse_url($baseUrl, PHP_URL_HOST) ?? '');
        $isLocalHost = in_array($host, ['localhost', '127.0.0.1', '::1'], true) || ($host !== '' && str_ends_with($host, '.test'));
        $appEnv = $isLocalHost ? 'local' : 'production';
        $appName = $this->domainToAppName($host) ?: (string) config('app.name', 'Societify');

        $envContent = $this->setOrAppendEnvValue($envContent, 'APP_URL', $baseUrl !== '' ? $baseUrl : 'http://localhost');
        $envContent = $this->setOrAppendEnvValue($envContent, 'APP_ENV', $appEnv);
        $envContent = $this->setOrAppendEnvValue($envContent, 'APP_DEBUG', 'false');
        $envContent = $this->setOrAppendEnvValue($envContent, 'APP_DEMO', 'false');
        $envContent = $this->setOrAppendEnvValue($envContent, 'APP_NAME', $this->quoteEnvValue($appName));

        // Avoid touching .env if nothing changed (prevents unnecessary dev-server reloads)
        if ($envContent === $originalEnvContent) {
            return;
        }

        file_put_contents($envPath, $envContent);

        // Keep the currently running process consistent for the rest of the installer flow
        config([
            'app.url' => $baseUrl,
            'app.env' => $appEnv,
            'app.debug' => false,
            'app.name' => $appName,
        ]);
    }

    /**
     * Quote an env value (useful for APP_NAME).
     */
    protected function quoteEnvValue(string $value): string
    {
        $escaped = str_replace('"', '\\"', $value);
        return '"' . $escaped . '"';
    }

    /**
     * Convert a domain host to a friendly app name.
     * Examples:
     * - societify.com         => Societify
     * - demo.societify.com    => Societify
     * - my-society.test       => My Society
     * - www.foo-bar.co.uk     => Foo Bar
     */
    protected function domainToAppName(string $host): string
    {
        $host = trim(strtolower($host));
        if ($host === '' || $host === 'localhost' || $host === '127.0.0.1' || $host === '::1') {
            return '';
        }

        $parts = array_values(array_filter(explode('.', $host), fn ($p) => $p !== ''));
        if (count($parts) === 0) {
            return '';
        }

        // Drop leading www
        if ($parts[0] === 'www') {
            array_shift($parts);
        }

        if (count($parts) === 1) {
            $label = $parts[0];
        } else {
            $tld = $parts[count($parts) - 1];
            $sld = $parts[count($parts) - 2];

            // Handle common multi-part TLDs like co.uk, com.au, co.in
            $commonSecondLevel = ['co', 'com', 'net', 'org', 'gov', 'edu'];
            if (strlen($tld) === 2 && in_array($sld, $commonSecondLevel, true) && count($parts) >= 3) {
                $label = $parts[count($parts) - 3];
            } else {
                $label = $sld;
            }
        }

        $label = preg_replace('/[^a-z0-9\-_]+/i', ' ', $label) ?? $label;
        $label = str_replace(['-', '_'], ' ', $label);
        $label = trim(preg_replace('/\s+/', ' ', $label) ?? $label);

        return $label !== '' ? ucwords($label) : '';
    }

    /**
     * Apply the chosen DB settings to the running process.
     *
     * Updating `.env` does not affect the current PHP process environment, so
     * we must set config() + reconnect before running migrations/seeders.
     */
    protected function applyDatabaseRuntimeConfig(string $connection, array $config): void
    {
        config(['database.default' => $connection]);

        if ($connection === 'sqlite') {
            config(['database.connections.sqlite.database' => database_path('database.sqlite')]);
        } else {
            config([
                'database.connections.mysql.host' => $config['db_host'] ?? '127.0.0.1',
                'database.connections.mysql.port' => $config['db_port'] ?? '3306',
                'database.connections.mysql.database' => $config['db_database'] ?? '',
                'database.connections.mysql.username' => $config['db_username'] ?? '',
                'database.connections.mysql.password' => $config['db_password'] ?? '',
            ]);
        }

        // Ensure Laravel drops any previous PDO connection.
        DB::purge();
        DB::reconnect();
    }

    /**
     * Set an env key value or append it if missing.
     */
    protected function setOrAppendEnvValue(string $envContent, string $key, string $value): string
    {
        $line = $key . '=' . $value;
        $pattern = '/^#?\s*' . preg_quote($key, '/') . '=.*/m';

        if (preg_match($pattern, $envContent) === 1) {
            return (string) preg_replace($pattern, $line, $envContent);
        }

        $suffix = str_ends_with($envContent, "\n") || $envContent === '' ? '' : "\n";
        return $envContent . $suffix . $line . "\n";
    }

    /**
     * Test MySQL connection
     */
    protected function testMySQLConnection(array $config): void
    {
        $pdo = new \PDO(
            "mysql:host={$config['db_host']};port={$config['db_port']};dbname={$config['db_database']}",
            $config['db_username'],
            $config['db_password'] ?? ''
        );
        $pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
    }

    /**
     * Create SQLite database file (deletes existing one for fresh install)
     */
    protected function createSQLiteDatabase(): void
    {
        $path = database_path('database.sqlite');
        
        // Ensure there are no open handles before replacing the file
        DB::purge('sqlite');
        
        // Delete existing database for fresh installation
        if (file_exists($path)) {
            unlink($path);
        }
        
        // Create new empty database file with proper permissions
        touch($path);
        chmod($path, 0666);
        
        // Verify the file was created
        if (!file_exists($path)) {
            throw new \Exception('Failed to create SQLite database file. Please check directory permissions.');
        }
        
        // Test the database connection by creating a simple table
        // Note: PDO::exec() is safe here as it's using hardcoded SQL strings, not user input
        // This is different from PHP's exec() function which executes system commands
        try {
            $pdo = new \PDO('sqlite:' . $path);
            $pdo->setAttribute(\PDO::ATTR_ERRMODE, \PDO::ERRMODE_EXCEPTION);
            // Test query to ensure database is writable
            // Using prepared statements would be overkill for this static test query
            $pdo->exec('CREATE TABLE IF NOT EXISTS test_table (id INTEGER PRIMARY KEY)');
            $pdo->exec('DROP TABLE test_table');
        } catch (\PDOException $e) {
            throw new \Exception('SQLite database file created but not writable: ' . $e->getMessage());
        }
    }

    /**
     * Run essential seeders for fresh installation
     */
    protected function runEssentialSeeders(): void
    {
        Artisan::call('db:seed', ['--class' => 'Database\\Seeders\\RoleSeeder', '--force' => true]);
        Artisan::call('db:seed', ['--class' => 'Database\\Seeders\\PermissionSeeder', '--force' => true]);
        Artisan::call('db:seed', ['--class' => 'Database\\Seeders\\RolePermissionSeeder', '--force' => true]);
        Artisan::call('db:seed', ['--class' => 'Database\\Seeders\\SubscriptionPlanSeeder', '--force' => true]);
        Artisan::call('db:seed', ['--class' => 'Database\\Seeders\\SystemSettingSeeder', '--force' => true]);
    }
}
