<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Http\Resources\MaintenanceTicketResource;
use App\Models\MaintenanceTicket;
use Illuminate\Support\Facades\Auth;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class VendorController extends Controller
{
    use AuthorizesRequests;

    /**
     * List tickets assigned to the authenticated vendor.
     */
    public function tickets(Request $request)
    {
        $user = Auth::user();
        
        $tickets = MaintenanceTicket::where('vendor_id', $user->id)
            ->where('society_id', $user->society_id)
            ->with(['user', 'user.unit'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return MaintenanceTicketResource::collection($tickets);
    }

    /**
     * Update ticket status and notes.
     */
    public function updateTicket(Request $request, MaintenanceTicket $ticket, \App\Services\FirebaseNotificationService $firebaseService)
    {
        $user = Auth::user();
        
        // Ensure ticket is assigned to this vendor
        if ($ticket->vendor_id !== $user->id || $ticket->society_id !== $user->society_id) {
            return response()->json([
                'message' => 'This ticket is not assigned to you.',
            ], 403);
        }
        
        $this->authorize('update', $ticket);

        $request->validate([
            'status' => 'required|in:assigned,in_progress,resolved',
            'resolution_notes' => 'nullable|string',
        ]);

        $oldStatus = $ticket->status;
        $newStatus = $request->status;

        $updateData = [
            'status' => $newStatus,
            'resolution_notes' => $request->resolution_notes,
        ];

        if ($newStatus === 'resolved' && ! $ticket->resolved_at) {
            $updateData['resolved_at'] = now();
        }

        $ticket->update($updateData);

        // Notify Resident
        if ($ticket->user && $ticket->user->fcm_token) {
            $firebaseService->sendToUser(
                $ticket->user, 
                'Ticket Status Updated', 
                "Your ticket #{$ticket->ticket_no} is now " . strtoupper(str_replace('_', ' ', $newStatus)),
                [
                    'type' => 'ticket_status',
                    'ticket_id' => (string) $ticket->id,
                    'status' => $newStatus,
                ]
            );
        }

        // Notify Admins
        $admins = \App\Models\User::where('society_id', $ticket->society_id)
            ->whereHas('role', fn($q) => $q->whereIn('name', ['society_admin', 'sub_admin']))
            ->get();
            
        if ($admins->isNotEmpty()) {
            $firebaseService->sendToUsers(
                $admins->all(), 
                'Ticket Status Updated', 
                "Ticket #{$ticket->ticket_no} updated to " . strtoupper(str_replace('_', ' ', $newStatus)) . " by Vendor", 
                [
                    'type' => 'ticket_status',
                    'ticket_id' => (string) $ticket->id,
                ]
            );
        }

        return new MaintenanceTicketResource($ticket);
    }
}
