<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class UserController extends Controller
{
    /**
     * Get all users with optional filters
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function index(Request $request): JsonResponse
    {
        $query = User::with(['role', 'society', 'unit.building']);

        // Filter by society_id
        if ($request->has('society_id')) {
            if ($request->get('society_id') === 'null') {
                $query->whereNull('society_id');
            } else {
                $query->where('society_id', $request->get('society_id'));
            }
        }

        // Filter by role_id
        if ($request->has('role_id') && $request->get('role_id') !== 'all') {
            $query->where('role_id', $request->get('role_id'));
        }

        // Filter by role name
        if ($request->has('role')) {
            $query->whereHas('role', function ($q) use ($request) {
                $q->where('name', $request->get('role'));
            });
        }

        // Filter by status
        if ($request->has('status') && $request->get('status') !== 'all') {
            $query->where('status', $request->get('status'));
        }

        // Filter by type (owner, tenant, family_member)
        if ($request->has('type')) {
            $query->where('type', $request->get('type'));
        }

        // Search by name, email, or phone
        if ($request->has('search')) {
            $query->search($request->get('search'));
        }

        // Filter active users only
        if ($request->boolean('active_only')) {
            $query->where('status', 'active');
        }

        // Filter web admins only
        if ($request->boolean('web_admins_only')) {
            $query->webAdmins();
        }

        // Filter mobile users only
        if ($request->boolean('mobile_users_only')) {
            $query->mobileUsers();
        }

        // Exclude deleted users
        $query->whereNull('deleted_at');

        // Ordering
        $orderBy = $request->get('order_by', 'created_at');
        $orderDir = $request->get('order_dir', 'desc');
        $query->orderBy($orderBy, $orderDir);

        // Pagination
        $perPage = $request->get('per_page', 15);
        
        if ($request->boolean('paginate', true)) {
            $users = $query->paginate($perPage)->withQueryString();
            
            return response()->json([
                'success' => true,
                'data' => $users->items(),
                'meta' => [
                    'current_page' => $users->currentPage(),
                    'last_page' => $users->lastPage(),
                    'per_page' => $users->perPage(),
                    'total' => $users->total(),
                    'from' => $users->firstItem(),
                    'to' => $users->lastItem(),
                ],
                'links' => [
                    'first' => $users->url(1),
                    'last' => $users->url($users->lastPage()),
                    'prev' => $users->previousPageUrl(),
                    'next' => $users->nextPageUrl(),
                ],
            ]);
        } else {
            $users = $query->get();
            
            return response()->json([
                'success' => true,
                'data' => $users,
                'count' => $users->count(),
            ]);
        }
    }

    /**
     * Get a specific user by ID
     *
     * @param int $id
     * @return JsonResponse
     */
    public function show(int $id): JsonResponse
    {
        $user = User::with(['role', 'society', 'unit.building', 'familyMembers', 'tenants'])
            ->findOrFail($id);

        return response()->json([
            'success' => true,
            'data' => $user,
        ]);
    }

    /**
     * Get users statistics
     *
     * @return JsonResponse
     */
    public function stats(): JsonResponse
    {
        $stats = [
            'total' => User::whereNull('deleted_at')->count(),
            'active' => User::where('status', 'active')->whereNull('deleted_at')->count(),
            'inactive' => User::where('status', 'inactive')->whereNull('deleted_at')->count(),
            'by_role' => User::selectRaw('roles.name as role_name, count(*) as count')
                ->join('roles', 'users.role_id', '=', 'roles.id')
                ->whereNull('users.deleted_at')
                ->groupBy('roles.name')
                ->pluck('count', 'role_name'),
            'by_type' => User::selectRaw('type, count(*) as count')
                ->whereNotNull('type')
                ->whereNull('deleted_at')
                ->groupBy('type')
                ->pluck('count', 'type'),
            'with_society' => User::whereNotNull('society_id')->whereNull('deleted_at')->count(),
            'without_society' => User::whereNull('society_id')->whereNull('deleted_at')->count(),
        ];

        return response()->json([
            'success' => true,
            'data' => $stats,
        ]);
    }
}
