<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Controllers\Api\RentRecordController;
use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class TenantController extends Controller
{
    /**
     * List all tenants.
     */
    public function index(Request $request)
    {
        $user = $request->user();

        if (! $user->isOwner()) {
            return response()->json(['message' => 'Only owners can manage tenants.'], 403);
        }

        $tenants = $user->tenants()->get();

        return response()->json(['data' => $tenants]);
    }

    /**
     * Add a new tenant.
     */
    public function store(Request $request)
    {
        $user = $request->user();

        if (! $user->isOwner()) {
            return response()->json(['message' => 'Only owners can add tenants.'], 403);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|regex:/^[0-9]{10}$/|unique:users,phone',
            'email' => 'nullable|email|unique:users,email',
            'agreement_end_date' => 'nullable|date|after:today',
            'monthly_rent' => 'nullable|numeric|min:0',
            'rent_increase_duration_months' => 'nullable|integer|min:1|max:120',
            'rent_increase_percentage' => 'nullable|numeric|min:0|max:100',
            'rent_agreement' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120', // 5MB
            'police_verification' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120',
        ]);

        $residentRole = Role::where('name', 'resident')->first();

        $tenant = User::create([
            'society_id' => $user->society_id,
            'role_id' => $residentRole->id,
            'parent_id' => $user->id,
            'type' => 'tenant',
            'name' => $request->name,
            'phone' => $request->phone,
            'email' => $request->email,
            'password' => Hash::make($request->phone),
            'unit_id' => $user->unit_id,
            'tenant_agreement_end_date' => $request->agreement_end_date,
            'monthly_rent' => $request->monthly_rent,
            'rent_increase_duration_months' => $request->rent_increase_duration_months,
            'rent_increase_percentage' => $request->rent_increase_percentage,
            'status' => 'active',
        ]);

        // Handle Documents
        if ($request->hasFile('rent_agreement')) {
            $path = $request->file('rent_agreement')->store('documents/tenants', 'public');
            $tenant->documents()->create([
                'type' => 'rent_agreement',
                'file_path' => $path,
                'valid_until' => $request->agreement_end_date, // Usually matches agrement
                'status' => 'pending',
            ]);
        }

        if ($request->hasFile('police_verification')) {
            $path = $request->file('police_verification')->store('documents/tenants', 'public');
            $tenant->documents()->create([
                'type' => 'police_verification',
                'file_path' => $path,
                'status' => 'pending',
            ]);
        }

        // Auto-generate rent records if rent info provided
        if ($tenant->monthly_rent && $tenant->tenant_agreement_end_date) {
            RentRecordController::generateRentRecords($tenant, $user);
        }

        return response()->json([
            'message' => 'Tenant added successfully.',
            'data' => $tenant->load('documents'),
        ], 201);
    }

    /**
     * Upload a document for an existing tenant.
     */
    public function uploadDocument(Request $request, $id)
    {
        $user = $request->user();

        // Check ownership
        if (! $user->isOwner()) {
             return response()->json(['message' => 'Unauthorized'], 403);
        }

        $tenant = $user->tenants()->where('id', $id)->first();
        if (! $tenant) {
            return response()->json(['message' => 'Tenant not found.'], 404);
        }

        $request->validate([
            'type' => 'required|string|in:rent_agreement,police_verification,national_id,other',
            'document' => 'required|file|mimes:pdf,jpg,jpeg,png|max:5120',
            'valid_until' => 'nullable|date',
        ]);

        $path = $request->file('document')->store('documents/tenants', 'public');

        $document = $tenant->documents()->create([
            'type' => $request->type,
            'file_path' => $path,
            'valid_until' => $request->valid_until,
            'status' => 'pending',
        ]);

        // If it's a rent agreement, optionally update the user's end date if provided
        if ($request->type === 'rent_agreement' && $request->valid_until) {
            $tenant->update(['tenant_agreement_end_date' => $request->valid_until]);
        }

        return response()->json([
            'message' => 'Document uploaded successfully.',
            'data' => $document,
        ]);
    }

    /**
     * Remove a tenant.
     */
    public function destroy(Request $request, $id)
    {
        $user = $request->user();

        if (! $user->isOwner()) {
            return response()->json(['message' => 'Only owners can remove tenants.'], 403);
        }

        $tenant = $user->tenants()->where('id', $id)->first();

        if (! $tenant) {
            return response()->json(['message' => 'Tenant not found.'], 404);
        }

        $tenant->delete();

        return response()->json(['message' => 'Tenant removed successfully.']);
    }
}
