<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\SubscriptionPayment;
use App\Models\SystemSetting;
use App\Services\PaymentGateway\PaymentGatewayFactory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class SubscriptionPaymentController extends Controller
{
    /**
     * Get available payment gateways for subscription payment.
     */
    public function getPaymentGateways(): JsonResponse
    {
        $gateways = PaymentGatewayFactory::getGatewayConfigs();

        if (empty($gateways)) {
            return response()->json([
                'success' => false,
                'message' => 'No payment gateways configured.',
            ], 400);
        }

        return response()->json([
            'success' => true,
            'gateways' => $gateways,
        ]);
    }

    /**
     * Create a payment order for subscription payment.
     * Supports multiple payment gateways.
     */
    public function createPaymentOrder(Request $request): JsonResponse
    {
        $request->validate([
            'gateway' => 'nullable|string|in:razorpay,stripe,paypal',
        ]);

        try {
            $user = Auth::user();
            
            if (!$user || !$user->society_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'User or society not found.',
                ], 400);
            }

            // Load society with subscription and plan relationships
            $society = $user->society()->with(['subscription.plan'])->first();
            
            if (!$society) {
                return response()->json([
                    'success' => false,
                    'message' => 'Society not found.',
                ], 400);
            }
            
            $subscription = $society->subscription;

            if (!$subscription) {
                // No subscription found - this should redirect to onboarding instead
                // But if they're already here, we need to handle it
                return response()->json([
                    'success' => false,
                    'message' => 'No subscription found for your society. Please complete the onboarding process first.',
                    'redirect_to_onboarding' => true,
                ], 400);
            }

            // Check if subscription is actually active and not expired
            // Allow renewal if subscription is expired (even if status is still 'active')
            // Also allow renewal if in trial or pending_payment state
            $canRenew = $subscription->isExpired() 
                || $subscription->status === 'trial' 
                || $subscription->status === 'pending_payment'
                || $subscription->status === 'expired';
            
            if (!$canRenew && $subscription->isActive()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Your subscription is already active and not expired.',
                ], 400);
            }

            // Get the payment gateway
            $gatewayId = $request->input('gateway');
            $gateway = $gatewayId 
                ? PaymentGatewayFactory::getGateway($gatewayId)
                : PaymentGatewayFactory::getDefaultGateway();

            if (!$gateway) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment gateway not configured. Please contact support.',
                ], 400);
            }

            // Calculate amount based on billing cycle
            // Try to get price from plan first, fallback to subscription fields
            $plan = $subscription->plan;
            
            if ($plan) {
                // Use plan prices if available
                $amount = match ($subscription->billing_cycle) {
                    'monthly' => $plan->monthly_price ?? 0,
                    'quarterly' => $plan->quarterly_price ?? ($plan->monthly_price * 3 ?? 0),
                    'annual' => $plan->annual_price ?? 0,
                    default => $plan->monthly_price ?? 0,
                };
            } else {
                // Fallback to subscription fields (backward compatibility)
                $amount = match ($subscription->billing_cycle) {
                    'monthly' => $subscription->monthly_price ?? 0,
                    'quarterly' => ($subscription->monthly_price ?? 0) * 3,
                    'annual' => $subscription->annual_price ?? 0,
                    default => $subscription->monthly_price ?? 0,
                };
            }

            if (!$amount || $amount <= 0) {
                Log::error('Invalid subscription amount', [
                    'subscription_id' => $subscription->id,
                    'billing_cycle' => $subscription->billing_cycle,
                    'plan_id' => $subscription->plan_id,
                    'amount_calculated' => $amount,
                ]);
                
                return response()->json([
                    'success' => false,
                    'message' => 'Invalid subscription amount. Please contact support.',
                ], 400);
            }

            // Create order using the gateway
            $orderData = $gateway->createOrder($amount, 'INR', [
                'receipt' => 'sub_activate_' . $subscription->id . '_' . uniqid(),
                'notes' => [
                    'subscription_id' => $subscription->id,
                    'society_id' => $society->id,
                    'user_id' => $user->id,
                    'type' => 'subscription_activation',
                ],
                'return_url' => url('/payment/success'),
                'cancel_url' => url('/payment/cancel'),
            ]);

            // Create pending payment record
            SubscriptionPayment::create([
                'subscription_id' => $subscription->id,
                'society_id' => $society->id,
                'razorpay_order_id' => $orderData['order_id'],
                'amount' => $amount,
                'currency' => 'INR',
                'status' => 'pending',
                'payment_gateway' => $gateway->getIdentifier(),
                'notes' => 'Subscription activation payment',
            ]);

            return response()->json([
                'success' => true,
                'gateway' => $gateway->getIdentifier(),
                'order_id' => $orderData['order_id'],
                'amount' => $amount,
                'currency' => 'INR',
                'key' => $orderData['key'] ?? null,
                'client_secret' => $orderData['client_secret'] ?? null,
                'approval_url' => $orderData['approval_url'] ?? null,
                'subscription' => [
                    'id' => $subscription->id,
                    'plan_name' => $subscription->plan_display_name ?? $subscription->plan_name,
                    'billing_cycle' => $subscription->billing_cycle,
                ],
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            Log::error('Subscription payment order creation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => Auth::id(),
                'society_id' => Auth::user()?->society_id,
                'subscription_id' => Auth::user()?->society?->subscription?->id,
                'subscription_status' => Auth::user()?->society?->subscription?->status,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Failed to create payment order: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Verify payment and activate subscription.
     * Supports multiple payment gateways.
     */
    public function verifyPayment(Request $request): JsonResponse
    {
        $request->validate([
            'gateway' => 'nullable|string|in:razorpay,stripe,paypal',
        ]);

        try {
            $user = Auth::user();
            
            if (!$user || !$user->society_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'User or society not found.',
                ], 400);
            }

            $society = $user->society;
            $subscription = $society->subscription;

            if (!$subscription) {
                return response()->json([
                    'success' => false,
                    'message' => 'No subscription found.',
                ], 400);
            }

            // Get the payment gateway
            // Try to get from request, or get from payment record, or use default
            $gatewayId = $request->input('gateway');
            
            // If gateway not provided, try to find it from pending payment
            if (!$gatewayId) {
                $orderId = $request->input('razorpay_order_id') ?? $request->input('order_id');
                if ($orderId) {
                    $pendingPayment = SubscriptionPayment::where('razorpay_order_id', $orderId)
                        ->where('subscription_id', $subscription->id)
                        ->first();
                    if ($pendingPayment && $pendingPayment->payment_gateway) {
                        $gatewayId = $pendingPayment->payment_gateway;
                    }
                }
            }
            
            // Still no gateway? Use default
            if (!$gatewayId) {
                $gateway = PaymentGatewayFactory::getDefaultGateway();
            } else {
                $gateway = PaymentGatewayFactory::getGateway($gatewayId);
            }

            if (!$gateway) {
                return response()->json([
                    'success' => false,
                    'message' => 'Payment gateway not configured.',
                ], 500);
            }

            // Verify payment using the gateway
            $result = $gateway->verifyPayment($request->all());

            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'message' => $result['error'] ?? 'Payment verification failed.',
                ], 400);
            }

            // Update payment record
            $orderId = $result['order_id'] ?? $request->input('order_id') ?? $request->input('razorpay_order_id');
            $payment = SubscriptionPayment::where('razorpay_order_id', $orderId)
                ->where('subscription_id', $subscription->id)
                ->first();

            if ($payment) {
                $payment->update([
                    'razorpay_payment_id' => $result['payment_id'],
                    'razorpay_signature' => $request->input('razorpay_signature'),
                    'status' => 'completed',
                    'paid_at' => now(),
                ]);
            } else {
                // Create payment record if not found
                SubscriptionPayment::create([
                    'subscription_id' => $subscription->id,
                    'society_id' => $society->id,
                    'razorpay_order_id' => $orderId,
                    'razorpay_payment_id' => $result['payment_id'],
                    'razorpay_signature' => $request->input('razorpay_signature'),
                    'amount' => $subscription->monthly_price ?? 0,
                    'currency' => 'INR',
                    'status' => 'completed',
                    'payment_gateway' => $gateway->getIdentifier(),
                    'paid_at' => now(),
                    'notes' => 'Subscription activation payment',
                ]);
            }

            // Activate subscription
            $startDate = now();
            $endDate = match ($subscription->billing_cycle) {
                'monthly' => $startDate->copy()->addMonth(),
                'quarterly' => $startDate->copy()->addMonths(3),
                'annual' => $startDate->copy()->addYear(),
                default => $startDate->copy()->addMonth(),
            };

            $subscription->update([
                'status' => 'active',
                'start_date' => $startDate,
                'end_date' => $endDate,
                'next_billing_date' => $endDate->copy(),
                'payment_gateway' => $gateway->getIdentifier(),
                'trial_start_date' => null,
                'trial_end_date' => null,
                'subscription_id' => $subscription->id,
                'society_id' => $society->id,
                'payment_id' => $result['payment_id'],
                'gateway' => $gateway->getIdentifier(),
            ]);

            // Subscription activated successfully via payment
            // No logging needed for successful operations

            // Send payment and activation emails
            $paymentRecord = $payment ?? SubscriptionPayment::where('razorpay_order_id', $orderId)->first();
            if ($paymentRecord) {
                \App\Services\EmailService::sendPaymentReceivedEmail($society, $subscription, $paymentRecord);
            }
            \App\Services\EmailService::sendSubscriptionActivatedEmail($society, $subscription);


            return response()->json([
                'success' => true,
                'message' => 'Payment verified and subscription activated successfully!',
            ]);
        } catch (\Exception $e) {
            // Log error for debugging payment verification issues
            Log::error('Subscription payment verification error', [
                'error' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage(),
            ], 500);
        }
    }
}
