<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SocietyController extends Controller
{
    /**
     * Get society information.
     */
    public function info(Request $request)
    {
        $user = Auth::user();
        $society = $user->society()->with(['settings', 'subscription'])->first();

        if (! $society) {
            return response()->json(['message' => 'Society not found'], 404);
        }

        $settings = $society->settings;

        return response()->json([
            // Basic Information
            'id' => $society->id,
            'name' => $society->name,
            'code' => $society->code,
            'registration_no' => $society->registration_no,
            
            // Contact & Address
            'address' => $society->address,
            'city' => $society->city,
            'state' => $society->state,
            'pincode' => $society->pincode,
            'phone' => $society->phone,
            'email' => $society->email,
            'full_address' => $society->full_address,
            
            // Branding
            'logo' => $society->logo ? asset('storage/' . $society->logo) : null,
            
            // Legal & Tax
            'gst_no' => $society->gst_no,
            'pan_no' => $society->pan_no,
            
            // Rules & Policies
            'rules' => $society->rules,
            'status' => $society->status,
            
            // Regional Settings
            'regional' => [
                'timezone' => $settings?->timezone ?? 'Asia/Kolkata',
                'currency' => $settings?->currency ?? 'INR',
                'date_format' => $settings?->date_format ?? 'Y-m-d',
                'time_format' => $settings?->time_format ?? 'H:i',
                'language' => $settings?->language ?? 'en',
            ],
            
            // Feature Flags
            'features' => [
                'visitor_management' => $settings?->enable_visitor_management ?? true,
                'maintenance_tickets' => $settings?->enable_maintenance_tickets ?? true,
                'complaints' => $settings?->enable_complaints ?? true,
                'events' => $settings?->enable_events ?? false,
                'documents' => $settings?->enable_documents ?? false,
                'directory' => $settings?->enable_directory ?? true,
                'notices' => $settings?->enable_notices ?? true,
                'polls' => $settings?->enable_polls ?? false,
                'marketplace' => $settings?->enable_marketplace ?? false,
            ],
            
            // Notification Preferences
            'notifications' => [
                'bill_generated' => $settings?->notify_bill_generated ?? true,
                'payment_received' => $settings?->notify_payment_received ?? true,
                'visitor_arrival' => $settings?->notify_visitor_arrival ?? true,
                'maintenance_request' => $settings?->notify_maintenance_request ?? true,
                'announcement' => $settings?->notify_announcement ?? true,
            ],
            
            // Communication Channels (Only status, no credentials)
            'communication_channels' => [
                'email' => $settings?->email_enabled ?? false,
                'sms' => $settings?->sms_enabled ?? false,
                'whatsapp' => $settings?->whatsapp_enabled ?? false,
                'push_notifications' => $settings?->fcm_enabled ?? false,
            ],
            
            // Security Settings
            'security' => [
                'require_otp_for_login' => $settings?->require_otp_for_login ?? false,
                'require_otp_for_payment' => $settings?->require_otp_for_payment ?? false,
                'enable_2fa' => $settings?->enable_2fa ?? false,
                'session_timeout_minutes' => $settings?->session_timeout_minutes ?? 60,
            ],
            
            // Payment Gateway (INCLUDES SECRET KEYS - Use with caution!)
            // WARNING: Never expose secret keys to clients.
            'payment_gateway' => [
                'active' => $settings?->payment_gateway ?? 'razorpay',
                'razorpay' => [
                    'enabled' => $settings?->razorpay_enabled ?? false,
                    'key_id' => $settings?->razorpay_enabled ? $settings?->razorpay_key_id : null,
                    'key_secret' => null,
                ],
                'stripe' => [
                    'enabled' => $settings?->stripe_enabled ?? false,
                    'public_key' => $settings?->stripe_enabled ? $settings?->stripe_public_key : null,
                    'secret_key' => null,
                ],
                'payu' => [
                    'enabled' => $settings?->payu_enabled ?? false,
                    'merchant_key' => $settings?->payu_enabled ? $settings?->payu_merchant_key : null,
                    'merchant_salt' => null,
                ],
                'cashfree' => [
                    'enabled' => $settings?->cashfree_enabled ?? false,
                    'app_id' => $settings?->cashfree_enabled ? $settings?->cashfree_app_id : null,
                    'secret_key' => null,
                ],
            ],
            
            // Subscription Info (if available)
            'subscription' => $society->subscription ? [
                'plan_name' => $society->subscription->plan_name ?? null,
                'status' => $society->subscription->status ?? null,
                'plan_expiry' => $society->subscription->plan_expiry ?? null,
                'max_users' => $society->subscription->max_users ?? null,
                'max_units' => $society->subscription->max_units ?? null,
                'max_buildings' => $society->subscription->max_buildings ?? null,
            ] : null,
        ]);
    }

    /**
     * Get payment gateway configuration (public keys only).
     */
    public function paymentConfig(Request $request)
    {
        $user = Auth::user();
        $settings = $user->society->settings;

        if (! $settings) {
            return response()->json(['message' => 'Settings not found'], 404);
        }

        // Use the factory to get available gateways
        $gateways = \App\Services\PaymentGateway\SocietyPaymentGatewayFactory::getGatewayConfigs($settings);

        return response()->json([
            'default_gateway' => $settings->payment_gateway ?? 'razorpay',
            'gateways' => $gateways,
            // Legacy format for backward compatibility
            'razorpay' => [
                'enabled' => $settings->razorpay_enabled ?? false,
                'key_id' => $settings->razorpay_enabled ? $settings->razorpay_key_id : null,
            ],
            'stripe' => [
                'enabled' => $settings->stripe_enabled ?? false,
                'public_key' => $settings->stripe_enabled ? $settings->stripe_public_key : null,
            ],
            'payu' => [
                'enabled' => $settings->payu_enabled ?? false,
                'merchant_key' => $settings->payu_enabled ? $settings->payu_merchant_key : null,
            ],
            'cashfree' => [
                'enabled' => $settings->cashfree_enabled ?? false,
                'app_id' => $settings->cashfree_enabled ? $settings->cashfree_app_id : null,
            ],
        ]);
    }

    /**
     * List buildings in the society with their units.
     */
    public function buildings(Request $request)
    {
        $user = Auth::user();
        
        $buildings = \App\Models\Building::where('society_id', $user->society_id)
            ->with(['units' => function($query) {
                // Determine what unit fields to select. Assuming standard fields + sorting
                $query->orderBy('unit_no');
            }])
            ->withCount('units')
            ->get();

        return response()->json([
            'data' => $buildings->map(function ($building) {
                return [
                    'id' => $building->id,
                    'name' => $building->name,
                    'building_no' => $building->building_no,
                    'floors' => $building->floors,
                    'total_units' => $building->total_units,
                    'address' => $building->address,
                    'units_count' => $building->units_count,
                    'units' => $building->units->map(function ($unit) {
                        return [
                            'id' => $unit->id,
                            'unit_no' => $unit->unit_no,
                            'floor' => $unit->floor,
                            'type' => $unit->type, // e.g., 2BHK, 3BHK
                            'status' => $unit->status, // occupied, vacant, etc.
                            'area' => $unit->area,
                        ];
                    }),
                ];
            }),
        ]);
    }
}
