<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Http\Resources\BillResource;
use App\Models\Bill;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;

class ResidentController extends Controller
{
    use AuthorizesRequests;

    /**
     * List all bills for the authenticated resident.
     */
    public function bills(Request $request)
    {
        $user = Auth::user();
        
        // For family members, get bills for the owner (parent) of their household
        // For tenants, get bills for the owner of their unit
        // For owners, get their own bills
        $billOwnerId = $user->id;
        
        if ($user->type === 'family_member' && $user->parent_id) {
            $billOwnerId = $user->parent_id;
        } elseif ($user->type === 'tenant' && $user->unit_id) {
            // Tenant: show owner's bills (owner of the same unit)
            $owner = \App\Models\User::where('unit_id', $user->unit_id)
                ->where(function ($q) {
                    $q->where('type', 'owner')
                      ->orWhereNull('type'); // Legacy owners might have null type
                })
                ->where('id', '!=', $user->id)
                ->first();
            
            if ($owner) {
                $billOwnerId = $owner->id;
            }
            // If no owner found, fallback to showing tenant's own bills (if any)
        }
        
        $bills = Bill::where('user_id', $billOwnerId)
            ->where('society_id', $user->society_id)
            ->with(['user.unit.building', 'society', 'billCycle', 'payment'])
            ->orderBy('due_date', 'desc')
            ->paginate(10);

        // Calculate Total Outstanding Amount
        $totalOutstanding = Bill::where('user_id', $billOwnerId)
            ->where('society_id', $user->society_id)
            ->whereIn('status', ['due', 'overdue', 'partial', 'approved'])
            ->sum('amount'); 

        // Refined Calculation
        $allOutstandingBills = Bill::where('user_id', $billOwnerId)
            ->where('society_id', $user->society_id)
            ->whereIn('status', ['due', 'overdue', 'partial', 'approved'])
            ->get();
            
        $realTotalDue = 0;
        foreach ($allOutstandingBills as $b) {
            $paid = \App\Models\Payment::where('bill_id', $b->id)->where('status', 'completed')->sum('amount');
            $due = $b->amount - $paid;
            if ($due > 0) {
                $realTotalDue += $due;
            }
        }

        return BillResource::collection($bills)->additional([
            'meta' => [
                'total_outstanding' => $realTotalDue,
                'total_pending_bills' => $allOutstandingBills->count(),
            ]
        ]);
    }

    /**
     * Show details of a specific bill.
     */
    public function showBill(Request $request, \App\Models\Bill $bill)
    {
        $user = Auth::user();
        
        // For family members, allow viewing bills of their parent/owner
        // For tenants, allow viewing bills of the owner of their unit
        // For owners, allow viewing their own bills
        $billOwnerId = $user->id;
        
        if ($user->type === 'family_member' && $user->parent_id) {
            $billOwnerId = $user->parent_id;
        } elseif ($user->type === 'tenant' && $user->unit_id) {
            $owner = \App\Models\User::where('unit_id', $user->unit_id)
                ->where(function ($q) {
                    $q->where('type', 'owner')
                      ->orWhereNull('type');
                })
                ->where('id', '!=', $user->id)
                ->first();
            
            if ($owner) {
                $billOwnerId = $owner->id;
            }
        }
        
        if ($bill->user_id !== $billOwnerId || $bill->society_id !== $user->society_id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }
        
        $bill->load(['user.unit.building', 'society', 'billCycle', 'payment']);

        return new \App\Http\Resources\BillResource($bill);
    }

    /**
     * List resident's maintenance tickets.
     * For family members and tenants, show tickets of their parent/owner.
     */
    public function tickets(Request $request)
    {
        $user = Auth::user();
        
        // For family members, get tickets of their parent/owner
        // For tenants, get tickets of the owner of their unit
        // For owners, get their own tickets
        $ticketOwnerId = $user->id;
        
        if ($user->type === 'family_member' && $user->parent_id) {
            // Family member: show parent's tickets
            $ticketOwnerId = $user->parent_id;
        } elseif ($user->type === 'tenant' && $user->unit_id) {
            // Tenant: show owner's tickets (owner of the same unit)
            $owner = \App\Models\User::where('unit_id', $user->unit_id)
                ->where(function ($q) {
                    $q->where('type', 'owner')
                      ->orWhereNull('type'); // Legacy owners might have null type
                })
                ->where('id', '!=', $user->id)
                ->first();
            
            if ($owner) {
                $ticketOwnerId = $owner->id;
            }
            // If no owner found, fallback to showing tenant's own tickets (if any)
        }
        
        $tickets = \App\Models\MaintenanceTicket::where('user_id', $ticketOwnerId)
            ->where('society_id', $user->society_id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return \App\Http\Resources\MaintenanceTicketResource::collection($tickets);
    }

    /**
     * Create a new maintenance ticket.
     */
    public function storeTicket(Request $request)
    {
        $request->validate([
            'category' => 'required|string',
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'priority' => 'required|in:low,medium,high,urgent',
            'is_urgent' => 'boolean',
            'images' => 'nullable|array', // Assuming array of base64 or URLs
        ]);

        $user = Auth::user();

        $ticket = \App\Models\MaintenanceTicket::create([
            'society_id' => $user->society_id,
            'user_id' => $user->id,
            'unit_id' => $user->unit_id,
            'ticket_no' => strtoupper('TKT-' . uniqid()),
            'category' => $request->category, // Ensure column exists or remove if not in DB
            'title' => $request->title,
            'description' => $request->description,
            'priority' => $request->priority,
            'status' => 'open',
            'is_urgent' => $request->is_urgent ?? false, // Ensure column exists
        ]);

        // Notify Admins
        try {
            $firebaseService = app(\App\Services\FirebaseNotificationService::class);
            $admins = \App\Models\User::where('society_id', $user->society_id)
                ->whereHas('role', fn($q) => $q->whereIn('name', ['society_admin', 'sub_admin']))
                ->get();
                
            if ($admins->isNotEmpty()) {
                $firebaseService->sendToUsers(
                    $admins->all(), 
                    'New Maintenance Ticket', 
                    "New ticket #{$ticket->ticket_no}: {$ticket->title} - {$ticket->priority}", 
                    [
                        'type' => 'new_ticket',
                        'ticket_id' => (string) $ticket->id,
                    ]
                );
            }
        } catch (\Exception $e) {
            // Log error for debugging notification failures (non-critical)
            \Illuminate\Support\Facades\Log::error('Failed to send new ticket notification: ' . $e->getMessage());
        }

        return new \App\Http\Resources\MaintenanceTicketResource($ticket);
    }

    /**
     * Show details of a specific ticket with history.
     */
    public function showTicket(Request $request, \App\Models\MaintenanceTicket $ticket)
    {
        $user = Auth::user();

        if ($ticket->user_id !== $user->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $ticket->load(['comments.user', 'timelines.user', 'vendor']);

        // Merge Timelines and Comments uniquely
        $history = collect();

        foreach ($ticket->timelines as $timeline) {
            $history->push([
                'type' => 'timeline', // 'timeline' or 'comment'
                'id' => $timeline->id,
                'action' => $timeline->action,
                'description' => $timeline->description,
                'timestamp' => $timeline->created_at,
                'user' => $timeline->user ? $timeline->user->name : 'System',
            ]);
        }

        foreach ($ticket->comments as $comment) {
            $history->push([
                'type' => 'comment',
                'id' => $comment->id,
                'message' => $comment->message,
                'images' => $comment->images,
                'timestamp' => $comment->created_at,
                'user' => $comment->user->name,
                'is_me' => $comment->user_id === $user->id,
            ]);
        }

        // Sort by timestamp descending (newest first) or ascending? 
        // Usually chat/history is ascending (oldest to newest) or descending.
        // Let's do descending (newest at top) for "Activity Feed". 
        // But chat is usually ascending.
        // User said "chronological list". Usually means oldest to newest.
        $sortedHistory = $history->sortByDesc('timestamp')->values();

        // Return Data (We can use a Resource, but merging history is custom)
        // Let's map it manually or extend resource.
        $data = (new \App\Http\Resources\MaintenanceTicketResource($ticket))->resolve();
        $data['history'] = $sortedHistory;

        return response()->json(['data' => $data]);
    }

    /**
     * List visitors for the resident's unit.
     * For family members and tenants, show visitors of their parent/owner.
     */
    public function visitors(Request $request)
    {
        $user = Auth::user();
        
        // For family members, get visitors of their parent/owner
        // For tenants, get visitors of the owner of their unit
        // For owners, get their own visitors
        $visitorOwnerId = $user->id;
        
        if ($user->type === 'family_member' && $user->parent_id) {
            // Family member: show parent's visitors
            $visitorOwnerId = $user->parent_id;
        } elseif ($user->type === 'tenant' && $user->unit_id) {
            // Tenant: show owner's visitors (owner of the same unit)
            $owner = \App\Models\User::where('unit_id', $user->unit_id)
                ->where(function ($q) {
                    $q->where('type', 'owner')
                      ->orWhereNull('type'); // Legacy owners might have null type
                })
                ->where('id', '!=', $user->id)
                ->first();
            
            if ($owner) {
                $visitorOwnerId = $owner->id;
            }
            // If no owner found, fallback to showing tenant's own visitors (if any)
        }
        
        $visitors = \App\Models\Visitor::where('user_id', $visitorOwnerId)
            ->where('society_id', $user->society_id)
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return \App\Http\Resources\VisitorResource::collection($visitors);
    }

    /**
     * Pre-approve a visitor.
     * For family members and tenants, create visitor for their parent/owner.
     */
    public function preApproveVisitor(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'type' => 'required|string|in:guest,delivery,service,cab',
            'expected_date' => 'required|date',
        ]);

        $user = Auth::user();
        
        // For family members, create visitor for parent/owner
        // For tenants, create visitor for owner of their unit
        // For owners, create visitor for themselves
        $visitorOwnerId = $user->id;
        
        if ($user->type === 'family_member' && $user->parent_id) {
            $visitorOwnerId = $user->parent_id;
        } elseif ($user->type === 'tenant' && $user->unit_id) {
            $owner = \App\Models\User::where('unit_id', $user->unit_id)
                ->where(function ($q) {
                    $q->where('type', 'owner')
                      ->orWhereNull('type');
                })
                ->where('id', '!=', $user->id)
                ->first();
            
            if ($owner) {
                $visitorOwnerId = $owner->id;
            }
        }

        $visitor = \App\Models\Visitor::create([
            'society_id' => $user->society_id,
            'user_id' => $visitorOwnerId,
            'name' => $request->name,
            'phone' => $request->phone,
            'type' => $request->type,
            'purpose' => $request->purpose ?? 'Visit', // Default purpose
            'status' => 'approved', // Pre-approved
            'gatepass_code' => strtoupper(substr(md5(uniqid()), 0, 8)),
            // 'expected_arrival' => $request->expected_date, // If column exists
        ]);

        return new \App\Http\Resources\VisitorResource($visitor);
    }

    /**
     * Approve a pending visitor.
     * For family members and tenants, allow managing visitors of their parent/owner.
     */
    public function approveVisitor(Request $request, \App\Models\Visitor $visitor)
    {
        $user = Auth::user();

        // Determine the owner ID (parent for family members, unit owner for tenants)
        $visitorOwnerId = $user->id;
        
        if ($user->type === 'family_member' && $user->parent_id) {
            $visitorOwnerId = $user->parent_id;
        } elseif ($user->type === 'tenant' && $user->unit_id) {
            $owner = \App\Models\User::where('unit_id', $user->unit_id)
                ->where(function ($q) {
                    $q->where('type', 'owner')
                      ->orWhereNull('type');
                })
                ->where('id', '!=', $user->id)
                ->first();
            
            if ($owner) {
                $visitorOwnerId = $owner->id;
            }
        }

        // Authorization: Ensure the visitor belongs to the owner (or user themselves)
        if ($visitor->user_id !== $visitorOwnerId && $visitor->user_id !== $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. You can only manage visitors for your unit.',
            ], 403);
        }

        // Check if visitor is in a status that can be approved
        if (!in_array($visitor->status, ['pending', 'waiting_approval'])) {
            return response()->json([
                'success' => false,
                'message' => 'Visitor cannot be approved. Current status: ' . $visitor->status,
            ], 422);
        }

        // Update visitor status to approved
        $visitor->update([
            'status' => 'approved',
        ]);

        // Notify Guard
        if ($visitor->checked_in_by) {
            $guard = \App\Models\User::find($visitor->checked_in_by);
            if ($guard && $guard->fcm_token) {
                $firebaseService = app(\App\Services\FirebaseNotificationService::class);
                $firebaseService->sendVisitorApprovedNotification($guard->fcm_token, $visitor->toArray());
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Visitor approved successfully',
            'data' => new \App\Http\Resources\VisitorResource($visitor->fresh()),
        ]);
    }

    /**
     * Reject a pending visitor.
     * For family members and tenants, allow managing visitors of their parent/owner.
     */
    public function rejectVisitor(Request $request, \App\Models\Visitor $visitor)
    {
        $user = Auth::user();

        // Determine the owner ID (parent for family members, unit owner for tenants)
        $visitorOwnerId = $user->id;
        
        if ($user->type === 'family_member' && $user->parent_id) {
            $visitorOwnerId = $user->parent_id;
        } elseif ($user->type === 'tenant' && $user->unit_id) {
            $owner = \App\Models\User::where('unit_id', $user->unit_id)
                ->where(function ($q) {
                    $q->where('type', 'owner')
                      ->orWhereNull('type');
                })
                ->where('id', '!=', $user->id)
                ->first();
            
            if ($owner) {
                $visitorOwnerId = $owner->id;
            }
        }

        // Authorization: Ensure the visitor belongs to the owner (or user themselves)
        if ($visitor->user_id !== $visitorOwnerId && $visitor->user_id !== $user->id) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized. You can only manage visitors for your unit.',
            ], 403);
        }

        // Check if visitor is in a status that can be rejected
        if (!in_array($visitor->status, ['pending', 'waiting_approval', 'approved'])) {
            return response()->json([
                'success' => false,
                'message' => 'Visitor cannot be rejected. Current status: ' . $visitor->status,
            ], 422);
        }

        // Optional: Get rejection reason from request
        $reason = $request->input('reason', null);

        // Update visitor status to rejected
        $visitor->update([
            'status' => 'rejected',
            'notes' => $reason ? "Rejected: {$reason}" : $visitor->notes,
        ]);

        // Notify Guard
        if ($visitor->checked_in_by) {
            $guard = \App\Models\User::find($visitor->checked_in_by);
            if ($guard && $guard->fcm_token) {
                $firebaseService = app(\App\Services\FirebaseNotificationService::class);
                $firebaseService->sendVisitorRejectedNotification($guard->fcm_token, $visitor->toArray());
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Visitor rejected successfully',
            'data' => new \App\Http\Resources\VisitorResource($visitor->fresh()),
        ]);
    }

    /**
     * List payment history for the resident.
     */
    public function payments(Request $request)
    {
        $user = Auth::user();
        
        // For family members, show payments for their parent/owner's bills
        // For owners, show their own payments
        $billOwnerId = $user->id;
        if ($user->type === 'family_member' && $user->parent_id) {
            $billOwnerId = $user->parent_id;
        }
        
        // Get payments made by the user OR payments for bills owned by the bill owner
        $payments = \App\Models\Payment::where('society_id', $user->society_id)
            ->where(function ($q) use ($user, $billOwnerId) {
                // Payments made by this user
                $q->where('user_id', $user->id)
                  // Or payments for bills owned by the bill owner (parent for family members)
                  ->orWhereHas('bill', function ($billQ) use ($billOwnerId) {
                      $billQ->where('user_id', $billOwnerId);
                  });
            })
            ->with(['bill'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return \App\Http\Resources\PaymentResource::collection($payments);
    }

    /**
     * Create Payment Order for Bill Payment.
     * Supports multiple payment gateways.
     */
    public function createOrder(Request $request)
    {
        $request->validate([
            'bill_id' => 'required|exists:bills,id',
            'amount' => 'required|numeric|min:1',
            'gateway' => 'nullable|string|in:razorpay,stripe,payu,cashfree',
        ]);

        $user = Auth::user();
        
        // For family members, allow paying bills of their parent/owner
        // For owners, allow paying their own bills
        $billOwnerId = $user->id;
        if ($user->type === 'family_member' && $user->parent_id) {
            $billOwnerId = $user->parent_id;
        }
        
        $bill = Bill::where('id', $request->bill_id)
            ->where('user_id', $billOwnerId)
            ->where('society_id', $user->society_id)
            ->first();

        if (!$bill) {
            return response()->json(['message' => 'Bill not found or unauthorized'], 404);
        }

        // Prevent payment if bill is already paid
        if ($bill->status === 'paid') {
            return response()->json([
                'success' => false,
                'message' => 'This bill has already been paid. You cannot make another payment.',
            ], 400);
        }

        // Double-check: Calculate total paid amount to ensure bill is really unpaid
        $totalPaid = \App\Models\Payment::where('bill_id', $bill->id)
            ->where('status', 'completed')
            ->sum('amount');
        
        if ($totalPaid >= $bill->amount) {
            // Update bill status if it's not already marked as paid
            $bill->update(['status' => 'paid', 'paid_at' => now()]);
            
            return response()->json([
                'success' => false,
                'message' => 'This bill has already been fully paid. Total paid: ₹' . number_format($totalPaid, 2),
            ], 400);
        }

        // Load credentials from Society Settings
        $society = $user->society;
        $society->load('settings');
        $settings = $society->settings;

        if (!$settings) {
            return response()->json(['message' => 'Society settings not configured'], 400);
        }

        // Get the gateway using factory
        $gatewayId = $request->input('gateway');
        $gateway = $gatewayId 
            ? \App\Services\PaymentGateway\SocietyPaymentGatewayFactory::getGateway($settings, $gatewayId)
            : \App\Services\PaymentGateway\SocietyPaymentGatewayFactory::getDefaultGateway($settings);

        if (!$gateway) {
            return response()->json(['message' => 'Online payment not enabled for this society'], 400);
        }

        try {
            $orderData = $gateway->createOrder($request->amount, 'INR', [
                'receipt' => 'bill_rcpt_' . $bill->id . '_' . time(),
                'notes' => [
                    'bill_id' => (string) $bill->id,
                    'user_id' => (string) $user->id,
                    'society_id' => (string) $society->id,
                    'type' => 'bill_payment'
                ]
            ]);

            return response()->json([
                'gateway' => $gateway->getIdentifier(),
                'order_id' => $orderData['order_id'],
                'currency' => 'INR',
                'amount' => $request->amount * 100,
                'key' => $orderData['key'] ?? null,
                'client_secret' => $orderData['client_secret'] ?? null,
            ]);

        } catch (\Exception $e) {
            return response()->json(['message' => 'Failed to create order: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Verify Payment for Bill.
     * Supports multiple payment gateways.
     */
    public function verifyPayment(Request $request)
    {
        $request->validate([
            'gateway' => 'required|string|in:razorpay,stripe,payu,cashfree',
            'bill_id' => 'required|exists:bills,id',
            'amount' => 'required|numeric',
            // Razorpay specific
            'razorpay_order_id' => 'required_if:gateway,razorpay|nullable|string',
            'razorpay_payment_id' => 'required_if:gateway,razorpay|nullable|string',
            'razorpay_signature' => 'required_if:gateway,razorpay|nullable|string',
            // Stripe specific
            'payment_intent_id' => 'required_if:gateway,stripe|nullable|string',
            // PayU specific
            'payu_txnid' => 'required_if:gateway,payu|nullable|string',
            'payu_mihpayid' => 'required_if:gateway,payu|nullable|string',
            'payu_status' => 'required_if:gateway,payu|nullable|string',
            'payu_hash' => 'required_if:gateway,payu|nullable|string',
            // Cashfree specific
            'cashfree_order_id' => 'required_if:gateway,cashfree|nullable|string',
        ]);

        $user = Auth::user();
        
        // For family members, allow paying bills of their parent/owner
        // For owners, allow paying their own bills
        $billOwnerId = $user->id;
        if ($user->type === 'family_member' && $user->parent_id) {
            $billOwnerId = $user->parent_id;
        }
        
        $bill = Bill::where('id', $request->bill_id)
            ->where('user_id', $billOwnerId)
            ->where('society_id', $user->society_id)
            ->first();

        if (!$bill) {
            return response()->json(['message' => 'Bill not found or unauthorized'], 404);
        }

        // Load credentials
        $society = $user->society;
        $society->load('settings');
        $settings = $society->settings;

        if (!$settings) {
            return response()->json(['message' => 'Configuration error'], 500);
        }

        // Get gateway
        $gateway = \App\Services\PaymentGateway\SocietyPaymentGatewayFactory::getGateway($settings, $request->gateway);

        if (!$gateway) {
            return response()->json(['message' => 'Payment gateway not configured'], 500);
        }

        try {
            // Verify payment using gateway
            $result = $gateway->verifyPayment($request->all());

            if (!$result['success']) {
                return response()->json(['message' => $result['error'] ?? 'Payment verification failed'], 400);
            }

            // Signature Valid - Record Payment
            $payment = \Illuminate\Support\Facades\DB::transaction(function () use ($user, $bill, $request, $society, $result, $gateway) {
                
                // 1. Create Payment Record
                $payment = \App\Models\Payment::create([
                    'society_id' => $society->id,
                    'user_id' => $user->id,
                    'bill_id' => $bill->id,
                    'amount' => $request->amount,
                    'payment_method' => 'online',
                    'status' => 'completed',
                    'payment_no' => 'PAY-ONL-' . time(),
                    'transaction_id' => $result['payment_id'] ?? $request->razorpay_payment_id,
                    'notes' => 'Online Bill Payment via ' . $gateway->getName(),
                ]);

                // 2. Update Bill Status
                $totalPaid = \App\Models\Payment::where('bill_id', $bill->id)
                    ->where('status', 'completed')
                    ->sum('amount');
                
                if ($totalPaid >= $bill->amount) {
                    $bill->update(['status' => 'paid', 'paid_at' => now()]);
                } else {
                    $bill->update(['status' => 'partial']);
                }

                // 3. Record Transaction in Ledger
                $primaryAccount = \App\Models\BankAccount::where('society_id', $society->id)
                    ->where('is_primary', true)
                    ->first();
                 
                if (!$primaryAccount) {
                    $primaryAccount = \App\Models\BankAccount::where('society_id', $society->id)->first();
                }

                if ($primaryAccount) {
                    $payment->update(['bank_account_id' => $primaryAccount->id]);
                    
                    app(\App\Services\TransactionService::class)->recordIncome(
                        $payment->amount,
                        $primaryAccount->id,
                        'bill_payment',
                        $payment,
                        'online',
                        'Online Bill Payment: ' . $payment->transaction_id,
                        $user->id
                    );
                }

                // Send multi-channel payment notification (FCM, SMS, WhatsApp, Email)
                \App\Services\BroadcastNotificationService::forSociety($society)
                    ->sendPaymentReceivedNotification($payment, $user);

                return $payment;
            });

            return response()->json([
                'success' => true,
                'message' => 'Payment verified successfully',
                'payment_id' => $payment->id
            ]);

        } catch (\Exception $e) {
            return response()->json(['message' => 'Error recording payment: ' . $e->getMessage()], 500);
        }
    }
}
