<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\RentRecord;
use App\Models\User;
use Illuminate\Http\Request;
use Carbon\Carbon;

class RentRecordController extends Controller
{
    /**
     * List rent records for tenant (tenant viewing their own records).
     */
    public function tenantIndex(Request $request)
    {
        $user = $request->user();

        if ($user->type !== 'tenant') {
            return response()->json(['message' => 'Only tenants can access this endpoint.'], 403);
        }

        // Only show current month's rent record for tenants
        $currentMonth = Carbon::now()->startOfMonth();
        
        $record = RentRecord::where('tenant_id', $user->id)
            ->where('month', $currentMonth)
            ->first();

        // If no record exists for current month, return empty array
        if (!$record) {
            return response()->json(['data' => []]);
        }

        return response()->json(['data' => [$record]]);
    }

    /**
     * List all rent records for owner (all tenants).
     */
    public function ownerIndex(Request $request)
    {
        $user = $request->user();

        if (!$user->isOwner()) {
            return response()->json(['message' => 'Only owners can access this endpoint.'], 403);
        }

        $records = RentRecord::where('owner_id', $user->id)
            ->with('tenant:id,name,phone,email')
            ->orderBy('month', 'desc')
            ->orderBy('tenant_id')
            ->get();

        return response()->json(['data' => $records]);
    }

    /**
     * List rent records for a specific tenant (owner viewing tenant's records).
     */
    public function tenantRecords(Request $request, $tenantId)
    {
        $user = $request->user();

        if (!$user->isOwner()) {
            return response()->json(['message' => 'Only owners can access this endpoint.'], 403);
        }

        $tenant = $user->tenants()->where('id', $tenantId)->first();
        if (!$tenant) {
            return response()->json(['message' => 'Tenant not found.'], 404);
        }

        $records = RentRecord::where('tenant_id', $tenantId)
            ->where('owner_id', $user->id)
            ->orderBy('month', 'desc')
            ->get();

        return response()->json(['data' => $records]);
    }

    /**
     * Create a new rent record.
     */
    public function store(Request $request)
    {
        $user = $request->user();

        if (!$user->isOwner()) {
            return response()->json(['message' => 'Only owners can create rent records.'], 403);
        }

        $request->validate([
            'tenant_id' => 'required|exists:users,id',
            'amount' => 'required|numeric|min:0',
            'month' => 'required|date|date_format:Y-m-d',
            'notes' => 'nullable|string|max:1000',
        ]);

        $tenant = $user->tenants()->where('id', $request->tenant_id)->first();
        if (!$tenant) {
            return response()->json(['message' => 'Tenant not found.'], 404);
        }

        // Ensure month is first day of month
        $month = Carbon::parse($request->month)->startOfMonth();

        // Check if record already exists
        $existing = RentRecord::where('tenant_id', $request->tenant_id)
            ->where('month', $month)
            ->first();

        if ($existing) {
            return response()->json(['message' => 'Rent record for this month already exists.'], 422);
        }

        $record = RentRecord::create([
            'tenant_id' => $request->tenant_id,
            'owner_id' => $user->id,
            'amount' => $request->amount,
            'month' => $month,
            'notes' => $request->notes,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Rent record created successfully.',
            'data' => $record,
        ], 201);
    }

    /**
     * Update a rent record.
     */
    public function update(Request $request, $id)
    {
        $user = $request->user();

        if (!$user->isOwner()) {
            return response()->json(['message' => 'Only owners can update rent records.'], 403);
        }

        $record = RentRecord::where('id', $id)
            ->where('owner_id', $user->id)
            ->first();

        if (!$record) {
            return response()->json(['message' => 'Rent record not found.'], 404);
        }

        $request->validate([
            'amount' => 'sometimes|numeric|min:0',
            'month' => 'sometimes|date|date_format:Y-m-d',
            'notes' => 'nullable|string|max:1000',
        ]);

        if ($request->has('month')) {
            $month = Carbon::parse($request->month)->startOfMonth();
            
            // Check for duplicate if month changed
            $existing = RentRecord::where('tenant_id', $record->tenant_id)
                ->where('month', $month)
                ->where('id', '!=', $id)
                ->first();

            if ($existing) {
                return response()->json(['message' => 'Rent record for this month already exists.'], 422);
            }

            $record->month = $month;
        }

        if ($request->has('amount')) {
            $record->amount = $request->amount;
        }

        if ($request->has('notes')) {
            $record->notes = $request->notes;
        }

        $record->save();

        return response()->json([
            'success' => true,
            'message' => 'Rent record updated successfully.',
            'data' => $record,
        ]);
    }

    /**
     * Mark rent record as paid.
     * Allows both owners and tenants to mark rent as paid.
     * Tenants can upload receipt image.
     */
    public function markPaid(Request $request, $id)
    {
        $user = $request->user();

        // Check if user is owner or tenant
        $isOwner = $user->isOwner() || $user->type === null;
        $isTenant = $user->type === 'tenant';

        if (!$isOwner && !$isTenant) {
            return response()->json(['message' => 'Unauthorized. Only owners or tenants can mark rent as paid.'], 403);
        }

        // Find the record - owners see their tenants' records, tenants see their own
        if ($isOwner) {
            $record = RentRecord::where('id', $id)
                ->where('owner_id', $user->id)
                ->first();
        } else {
            // Tenant can only mark their own rent as paid
            $record = RentRecord::where('id', $id)
                ->where('tenant_id', $user->id)
                ->first();
        }

        if (!$record) {
            return response()->json(['message' => 'Rent record not found.'], 404);
        }

        if ($record->is_paid) {
            return response()->json(['message' => 'Rent record is already marked as paid.'], 422);
        }

        $request->validate([
            'payment_method' => 'nullable|string|max:50',
            'notes' => 'nullable|string|max:1000',
            'receipt_image' => 'nullable|image|mimes:jpeg,png,jpg|max:5120', // 5MB max
        ]);

        $receiptPath = null;
        if ($request->hasFile('receipt_image')) {
            $receiptPath = $request->file('receipt_image')->store('rent_receipts', 'public');
        } elseif ($request->has('receipt_image') && is_string($request->receipt_image)) {
            // Handle base64 encoded image
            try {
                $imageData = $request->receipt_image;
                if (preg_match('/^data:image\/(\w+);base64,/', $imageData, $type)) {
                    $imageData = substr($imageData, strpos($imageData, ',') + 1);
                    $type = strtolower($type[1]);
                    
                    if (!in_array($type, ['jpg', 'jpeg', 'png'])) {
                        return response()->json(['message' => 'Invalid image type. Only JPEG and PNG are allowed.'], 422);
                    }
                    
                    $imageData = base64_decode($imageData);
                    if ($imageData === false) {
                        return response()->json(['message' => 'Invalid base64 image data.'], 422);
                    }
                    
                    $filename = 'rent_' . $id . '_' . time() . '.' . $type;
                    $path = 'rent_receipts/' . $filename;
                    \Storage::disk('public')->put($path, $imageData);
                    $receiptPath = $path;
                }
            } catch (\Exception $e) {
                \Log::error('Error processing receipt image: ' . $e->getMessage());
                return response()->json(['message' => 'Failed to process receipt image.'], 422);
            }
        }

        // If tenant is paying, set status to pending_approval. If owner, set to paid directly.
        if ($isTenant) {
            $record->update([
                'payment_status' => 'pending_approval',
                'payment_method' => $request->payment_method ?? 'manual',
                'notes' => $request->notes ?? $record->notes,
                'receipt_image' => $receiptPath,
            ]);
            $message = 'Payment submitted successfully. Waiting for owner approval.';
        } else {
            // Owner marking as paid directly
            $record->update([
                'is_paid' => true,
                'payment_status' => 'paid',
                'paid_date' => now(),
                'payment_method' => $request->payment_method ?? $record->payment_method,
                'notes' => $request->notes ?? $record->notes,
                'receipt_image' => $receiptPath,
            ]);
            $message = 'Rent marked as paid successfully.';
        }

        return response()->json([
            'success' => true,
            'message' => $message,
            'data' => $record->fresh(),
        ]);
    }

    /**
     * Approve or reject tenant payment (owner only).
     */
    public function approvePayment(Request $request, $id)
    {
        $user = $request->user();

        if (!$user->isOwner() && $user->type !== null) {
            return response()->json(['message' => 'Only owners can approve payments.'], 403);
        }

        $record = RentRecord::where('id', $id)
            ->where('owner_id', $user->id)
            ->first();

        if (!$record) {
            return response()->json(['message' => 'Rent record not found.'], 404);
        }

        if ($record->payment_status !== 'pending_approval') {
            return response()->json(['message' => 'This payment is not pending approval.'], 422);
        }

        $request->validate([
            'action' => 'required|in:approve,reject',
            'notes' => 'nullable|string|max:1000',
        ]);

        if ($request->action === 'approve') {
            $record->update([
                'is_paid' => true,
                'payment_status' => 'paid',
                'paid_date' => now(),
                'notes' => $request->notes ?? $record->notes,
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Payment approved successfully.',
                'data' => $record->fresh(),
            ]);
        } else {
            // Reject payment
            $record->update([
                'payment_status' => 'pending',
                'notes' => ($request->notes ?? $record->notes) . ' (Payment rejected)',
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Payment rejected.',
                'data' => $record->fresh(),
            ]);
        }
    }

    /**
     * Get recent rent transactions for dashboard (owner or tenant view).
     */
    public function recentTransactions(Request $request)
    {
        $user = $request->user();
        $limit = $request->input('limit', 5);

        $query = RentRecord::query();

        $isOwner = $user->isOwner() || $user->type === null;
        
        if ($isOwner) {
            // Owner sees all their tenants' rent records
            $query->where('owner_id', $user->id)
                ->with('tenant:id,name,phone,email');
        } elseif ($user->type === 'tenant') {
            // Tenant sees only their own records
            $query->where('tenant_id', $user->id);
        } else {
            return response()->json(['data' => []]);
        }

        $transactions = $query->orderBy('updated_at', 'desc')
            ->orderBy('paid_date', 'desc')
            ->orderBy('month', 'desc')
            ->limit($limit)
            ->get()
            ->map(function ($record) use ($user, $isOwner) {
                $tenantName = null;
                
                if ($isOwner && $record->relationLoaded('tenant')) {
                    $tenantName = $record->tenant->name ?? 'Unknown';
                } elseif ($isOwner && $record->tenant_id) {
                    // Load tenant if not already loaded
                    $tenant = $record->tenant;
                    $tenantName = $tenant ? $tenant->name : 'Unknown';
                }

                return [
                    'id' => $record->id,
                    'type' => 'rent',
                    'title' => $isOwner && $tenantName 
                        ? ('Rent - ' . $tenantName)
                        : 'Rent Payment',
                    'amount' => (float) $record->amount,
                    'status' => $record->is_paid ? 'paid' : 'pending',
                    'date' => $record->paid_date ?? $record->month,
                    'month' => $record->month->format('Y-m-d'),
                    'month_display' => $record->month->format('M Y'),
                    'is_paid' => $record->is_paid,
                    'paid_date' => $record->paid_date ? $record->paid_date->toDateTimeString() : null,
                    'payment_method' => $record->payment_method,
                    'tenant_name' => $tenantName,
                ];
            });

        return response()->json(['data' => $transactions]);
    }

    /**
     * Helper: Generate rent records for a tenant based on agreement and rent settings.
     * This can be called when tenant is created or agreement is updated.
     */
    public static function generateRentRecords(User $tenant, User $owner)
    {
        if (!$tenant->monthly_rent || !$tenant->tenant_agreement_end_date) {
            return;
        }

        $startDate = $tenant->created_at ?? now();
        $endDate = $tenant->tenant_agreement_end_date;
        $currentRent = $tenant->monthly_rent;
        $increaseDuration = $tenant->rent_increase_duration_months ?? 12; // Default 12 months
        $increasePercentage = $tenant->rent_increase_percentage ?? 0;

        $month = Carbon::parse($startDate)->startOfMonth();
        $endMonth = Carbon::parse($endDate)->startOfMonth();
        $monthsSinceStart = 0;

        while ($month->lte($endMonth)) {
            // Check if record already exists
            $existing = RentRecord::where('tenant_id', $tenant->id)
                ->where('month', $month)
                ->first();

            if (!$existing) {
                // Calculate rent with periodic increases
                $rentIncreaseCount = floor($monthsSinceStart / $increaseDuration);
                $calculatedRent = $currentRent;
                
                for ($i = 0; $i < $rentIncreaseCount; $i++) {
                    $calculatedRent = $calculatedRent * (1 + ($increasePercentage / 100));
                }

                RentRecord::create([
                    'tenant_id' => $tenant->id,
                    'owner_id' => $owner->id,
                    'amount' => round($calculatedRent, 2),
                    'month' => $month->copy(),
                    'is_paid' => false,
                ]);
            }

            $month->addMonth();
            $monthsSinceStart++;
        }
    }
}
