<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Http\Resources\VisitorResource;
use App\Models\User;
use App\Models\Visitor;
use Illuminate\Support\Facades\Auth;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class GuardController extends Controller
{
    use AuthorizesRequests;

    /**
     * List visitors (History or Active based on filter).
     */
    public function visitors(Request $request)
    {
        $user = Auth::user();
        
        $query = Visitor::where('society_id', $user->society_id)
            ->with(['user', 'user.unit']);

        if ($request->has('status')) {
            $query->where('status', $request->status);
        } else {
            // If no status specified, return history (all except draft/cancelled maybe?)
            // Or just return all sorted by recent.
        }

        $visitors = $query->orderBy('created_at', 'desc')->paginate(20);

        return VisitorResource::collection($visitors);
    }

    /**
     * List expected visitors for today.
     */
    public function expectedVisitors(Request $request)
    {
        $user = Auth::user();
        
        // Assuming 'approved' status means pre-approved/expected
        // We might want to filter by date if expected_date exists, but keeping it simple for now
        $visitors = Visitor::where('society_id', $user->society_id)
            ->where('status', 'approved')
            ->with(['user', 'user.unit'])
            ->orderBy('created_at', 'asc') // Oldest approvals first (expected earlier)
            ->get();

        return VisitorResource::collection($visitors)->additional([
            'success' => true,
        ]);
    }

    /**
     * List verified visitors (currently inside, ready for checkout).
     */
    public function verifiedVisitors(Request $request)
    {
        $user = Auth::user();

        $visitors = Visitor::where('society_id', $user->society_id)
            ->where('status', 'checked_in')
            ->with(['user', 'user.unit'])
            ->orderBy('entry_time', 'desc')
            ->get();

        return VisitorResource::collection($visitors)->additional([
            'success' => true,
        ]);
    }

    /**
     * Verify Visitor OTP (Gatepass Code).
     */
    public function verifyVisitorOtp(Request $request)
    {
        $request->validate([
            'otp' => 'required|string',
        ]);

        $user = Auth::user();
        $otp = strtoupper(trim((string) $request->otp));

        $visitor = Visitor::where('society_id', $user->society_id)
            ->where('gatepass_code', $otp)
            ->with(['user', 'user.unit'])
            ->first();

        if (!$visitor) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid OTP or Visitor not found.',
            ], 404);
        }

        // Check if already checked out or rejected
        if (in_array($visitor->status, ['rejected', 'checked_out', 'cancelled'])) {
            return response()->json([
                'success' => false,
                'message' => 'This code is no longer valid. Status: ' . $visitor->status,
            ], 422);
        }

        // Mark as checked_in if not already
        if ($visitor->status !== 'checked_in') {
            $visitor->status = 'checked_in';
            $visitor->entry_time = now();
            $visitor->save();
        }
        
        $visitor->refresh();

        return response()->json([
            'success' => true,
            'message' => 'OTP Verified',
            'visitor' => new VisitorResource($visitor),
        ]);
    }

    /**
     * Check in a new visitor.
     */
    public function checkIn(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|max:20',
            'vehicle_number' => 'nullable|string|max:20',
            'type' => 'required|string|in:guest,delivery,service,staff,cab',
            'purpose' => 'nullable|string|max:255',
            // Support both payload shapes:
            // - older clients: resident_id
            // - mobile guard flow: unit_id (select flat/unit)
            'resident_id' => 'required_without:unit_id|nullable|exists:users,id',
            'unit_id' => 'required_without:resident_id|nullable|exists:units,id',
        ]);

        $user = Auth::user();

        // Resolve resident for this entry
        $resident = null;
        if ($request->filled('resident_id')) {
            $resident = User::where('id', $request->resident_id)
                ->where('society_id', $user->society_id)
                ->firstOrFail();
        } else {
            $unitId = (int) $request->unit_id;

            // Prefer the owner in that unit, else fallback to any user in that unit.
            $resident = User::where('society_id', $user->society_id)
                ->where('unit_id', $unitId)
                ->where(function ($q) {
                    $q->where('type', 'owner')
                      ->orWhereNull('type'); // legacy owners may have null type
                })
                ->first();

            if (! $resident) {
                $resident = User::where('society_id', $user->society_id)
                    ->where('unit_id', $unitId)
                    ->first();
            }

            if (! $resident) {
                return response()->json([
                    'success' => false,
                    'message' => 'No resident found for selected unit. Please assign a resident to that unit first.',
                ], 422);
            }
        }

        $visitor = Visitor::create([
            'society_id' => $user->society_id,
            'user_id' => $resident->id,
            'name' => $request->name,
            'phone' => $request->phone,
            'vehicle_number' => $request->vehicle_number,
            'type' => strtolower((string) $request->type),
            'purpose' => $request->purpose,
            'status' => 'pending', // Changed to pending until resident approves
            'entry_time' => now(),
            'checked_in_by' => $user->id,
            'gatepass_code' => strtoupper(substr(md5(uniqid()), 0, 8)),
        ]);

        // Send push notification to resident
        try {
            $firebaseService = app(\App\Services\FirebaseNotificationService::class);
            $firebaseService->sendVisitorArrivalNotification($resident, [
                'id' => $visitor->id,
                'name' => $visitor->name,
                'phone' => $visitor->phone,
                'type' => $visitor->type,
                'gatepass_code' => $visitor->gatepass_code, // Pass OTP
            ]);
        } catch (\Exception $e) {
            // Log warning for debugging notification failures (non-critical)
            \Illuminate\Support\Facades\Log::warning('Failed to send visitor notification: ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Visitor entry recorded',
            'data' => new VisitorResource($visitor),
        ]);
    }

    /**
     * Check out a visitor.
     */
    public function checkOut(Request $request, Visitor $visitor)
    {
        $this->authorize('checkOut', $visitor);

        $visitor->update([
            'status' => 'checked_out',
            'exit_time' => now(),
        ]);

        return new VisitorResource($visitor);
    }
}
