<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class FamilyMemberController extends Controller
{
    /**
     * List all family members.
     */
    public function index(Request $request)
    {
        $user = $request->user();

        // If user is a family member, they might want to see other family members or just the owner.
        // For now, assume this is for the Owner to list their family.
        if (! $user->isOwner()) {
            // Alternatively, allow family members to see their siblings/parents?
            // "access tha ll this flat or unit info to them"
            // For now, strict check: Only owner manages family.
            return response()->json(['message' => 'Only owners can manage family members.'], 403);
        }

        $members = $user->familyMembers()->get();

        return response()->json(['data' => $members]);
    }

    /**
     * Add a new family member.
     */
    public function store(Request $request)
    {
        $user = $request->user();

        if (! $user->isOwner()) {
            return response()->json(['message' => 'Only owners can add family members.'], 403);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'phone' => 'required|string|regex:/^[0-9]{10}$/|unique:users,phone',
            'email' => 'nullable|email|unique:users,email',
        ]);

        $residentRole = Role::where('name', 'resident')->first();

        $familyMember = User::create([
            'society_id' => $user->society_id,
            'role_id' => $residentRole->id,
            'parent_id' => $user->id,
            'type' => 'family_member',
            'name' => $request->name,
            'phone' => $request->phone,
            'email' => $request->email,
            'password' => Hash::make($request->phone), // Default password is phone, they should change it or use OTP
            'unit_id' => $user->unit_id,
            'status' => 'active',
        ]);

        return response()->json([
            'message' => 'Family member added successfully.',
            'data' => $familyMember,
        ], 201);
    }

    /**
     * Remove a family member.
     */
    public function destroy(Request $request, $id)
    {
        $user = $request->user();

        if (! $user->isOwner()) {
            return response()->json(['message' => 'Only owners can remove family members.'], 403);
        }

        $member = $user->familyMembers()->where('id', $id)->first();

        if (! $member) {
            return response()->json(['message' => 'Family member not found.'], 404);
        }

        $member->delete();

        return response()->json(['message' => 'Family member removed successfully.']);
    }
}
