<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Facility;
use App\Models\FacilitySlot;
use App\Models\FacilityBooking;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class FacilityController extends Controller
{
    /**
     * Display a listing of facilities for the society.
     */
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        $facilities = Facility::where('society_id', $user->society_id)
            ->with(['slots' => function ($query) {
                $query->where('is_active', true);
            }])
            ->when($request->is_active !== null, function ($query) use ($request) {
                $query->where('is_active', $request->is_active);
            })
            ->get();

        return response()->json([
            'success' => true,
            'facilities' => $facilities,
        ]);
    }

    /**
     * Store a newly created facility (Admin only).
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'images' => 'nullable|array',
            'images.*' => 'url',
            'capacity' => 'nullable|integer|min:1',
            'amenities' => 'nullable|string',
            'rules' => 'nullable|string',
            'cancellation_policy' => 'nullable|string',
            'requires_approval' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        $facility = Facility::create([
            'society_id' => $user->society_id,
            'name' => $request->name,
            'description' => $request->description,
            'images' => $request->images,
            'capacity' => $request->capacity,
            'amenities' => $request->amenities,
            'rules' => $request->rules,
            'cancellation_policy' => $request->cancellation_policy,
            'requires_approval' => $request->requires_approval ?? false,
            'is_active' => true,
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Facility created successfully',
            'facility' => $facility,
        ], 201);
    }

    /**
     * Display the specified facility.
     */
    public function show(Request $request, $id): JsonResponse
    {
        $user = $request->user();

        $facility = Facility::where('society_id', $user->society_id)
            ->with('slots')
            ->find($id);

        if (!$facility) {
            return response()->json([
                'success' => false,
                'message' => 'Facility not found',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'facility' => $facility,
        ]);
    }

    /**
     * Update the specified facility (Admin only).
     */
    public function update(Request $request, $id): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'images' => 'nullable|array',
            'images.*' => 'url',
            'capacity' => 'nullable|integer|min:1',
            'amenities' => 'nullable|string',
            'rules' => 'nullable|string',
            'cancellation_policy' => 'nullable|string',
            'requires_approval' => 'boolean',
            'is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        $facility = Facility::where('society_id', $user->society_id)->find($id);

        if (!$facility) {
            return response()->json([
                'success' => false,
                'message' => 'Facility not found',
            ], 404);
        }

        $facility->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Facility updated successfully',
            'facility' => $facility,
        ]);
    }

    /**
     * Remove the specified facility (Admin only).
     */
    public function destroy(Request $request, $id): JsonResponse
    {
        $user = $request->user();

        $facility = Facility::where('society_id', $user->society_id)->find($id);

        if (!$facility) {
            return response()->json([
                'success' => false,
                'message' => 'Facility not found',
            ], 404);
        }

        // Check if there are active bookings
        $hasActiveBookings = $facility->bookings()
            ->whereIn('booking_status', ['pending', 'confirmed'])
            ->exists();

        if ($hasActiveBookings) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete facility with active bookings',
            ], 400);
        }

        $facility->delete();

        return response()->json([
            'success' => true,
            'message' => 'Facility deleted successfully',
        ]);
    }

    /**
     * Get slots for a specific facility with availability for a date.
     */
    public function getSlots(Request $request, $facilityId) {
        $date = $request->query('date'); // Format: YYYY-MM-DD
        
        // 1. Validate date is present
        if (!$date) {
            return response()->json(['success' => false, 'message' => 'Date is required'], 400);
        }

        $facility = Facility::with(['slots'])->findOrFail($facilityId);

        // 2. Fetch slots and count bookings for EACH slot on THIS date
        $slots = $facility->slots->map(function ($slot) use ($date) {
            // Count confirmed OR pending bookings (blocked slots)
            $bookedCount = FacilityBooking::where('slot_id', $slot->id)
                ->whereDate('booking_date', $date)
                ->whereIn('booking_status', ['confirmed', 'pending']) 
                ->where('payment_status', '!=', 'failed')
                ->count();

            // 3. Inject the count into the response
            $slot->booked_count = $bookedCount;
            
            // 4. Determine if available (Frontend uses this too, but mainly booked_count)
            $slot->is_available = $slot->is_active && ($bookedCount < $slot->max_bookings_per_slot);
            
            return $slot;
        });

        return response()->json([
            'success' => true,
            'data' => $slots
        ]);
    }

    /**
     * Check bulk availability for a date range.
     */
    public function checkAvailability(Request $request, $id)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        $startDate = $request->start_date;
        $endDate = $request->end_date;

        $facility = Facility::where('is_active', true)->find($id);

        if (!$facility) {
            return response()->json([
                'success' => false,
                'message' => 'Facility not found',
            ], 404);
        }

        $slots = FacilitySlot::where('facility_id', $id)->where('is_active', true)->get();
        if ($slots->isEmpty()) {
             // If no slots, effectively unavailable or handled as such
             // But if facility active but no slots, maybe 'unavailable'?
             // Let's assume 'unavailable' if no slots defined.
             // Or iterate dates and say 'unavailable'.
        }

        // Fetch all bookings in range
        $bookings = FacilityBooking::where('facility_id', $id)
            ->whereBetween('booking_date', [$startDate, $endDate])
            ->whereIn('booking_status', ['confirmed', 'pending'])
            ->where('payment_status', '!=', 'failed')
            ->get()
            ->groupBy('booking_date');

        $period = \Carbon\CarbonPeriod::create($startDate, $endDate);
        $availability = [];

        foreach ($period as $date) {
            $dateStr = $date->format('Y-m-d');
            
            if (!$facility->is_active) {
                $availability[$dateStr] = 'unavailable';
                continue;
            }

            // Bookings for this specific date
            $dayBookings = isset($bookings[$dateStr]) ? $bookings[$dateStr] : collect([]);
            
            $allSlotsFull = true;
            $anySlotFull = false;
            $allSlotsAvailable = true;

            if ($slots->isEmpty()) {
                $availability[$dateStr] = 'unavailable';
                continue;
            }

            foreach ($slots as $slot) {
                $slotBookingsCount = $dayBookings->where('slot_id', $slot->id)->count();
                $isSlotFull = $slotBookingsCount >= $slot->max_bookings_per_slot;

                if (!$isSlotFull) {
                    $allSlotsFull = false;
                } else {
                    $anySlotFull = true;
                    $allSlotsAvailable = false;
                }
            }

            if ($allSlotsFull) {
                $status = 'full';
            } elseif ($anySlotFull) {
                $status = 'partial';
            } else {
                $status = 'available';
            }

            // Prepare slot details for this day
            $slotDetails = $slots->map(function($slot) use ($dayBookings) {
                $confirmedCount = $dayBookings->where('slot_id', $slot->id)->count();
                $isFull = $confirmedCount >= $slot->max_bookings_per_slot;
                
                return [
                    'id' => $slot->id,
                    'name' => $slot->name,
                    'is_booked' => $isFull, // Simplified for UI usage
                    'booked_count' => $confirmedCount,
                    'max_bookings' => $slot->max_bookings_per_slot 
                ];
            });

            $availability[$dateStr] = [
                'status' => $status,
                'slots' => $slotDetails
            ];
        }

        return response()->json([
            'success' => true,
            'data' => $availability,
        ]);
    }

    /**
     * Create or update slots for a facility (Admin only).
     */
    public function manageSlots(Request $request, $id): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'slots' => 'required|array',
            'slots.*.id' => 'nullable|exists:facility_slots,id',
            'slots.*.name' => 'required|string|max:255',
            'slots.*.start_time' => 'required|date_format:H:i',
            'slots.*.end_time' => 'required|date_format:H:i|after:slots.*.start_time',
            'slots.*.price' => 'required|numeric|min:0',
            'slots.*.is_active' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        $facility = Facility::where('society_id', $user->society_id)->find($id);

        if (!$facility) {
            return response()->json([
                'success' => false,
                'message' => 'Facility not found',
            ], 404);
        }

        $slots = [];

        foreach ($request->slots as $slotData) {
            if (isset($slotData['id'])) {
                // Update existing slot
                $slot = FacilitySlot::where('facility_id', $id)
                    ->find($slotData['id']);
                
                if ($slot) {
                    $slot->update([
                        'name' => $slotData['name'],
                        'start_time' => $slotData['start_time'],
                        'end_time' => $slotData['end_time'],
                        'price' => $slotData['price'],
                        'is_active' => $slotData['is_active'] ?? true,
                    ]);
                    $slots[] = $slot;
                }
            } else {
                // Create new slot
                $slot = FacilitySlot::create([
                    'facility_id' => $id,
                    'name' => $slotData['name'],
                    'start_time' => $slotData['start_time'],
                    'end_time' => $slotData['end_time'],
                    'price' => $slotData['price'],
                    'is_active' => $slotData['is_active'] ?? true,
                ]);
                $slots[] = $slot;
            }
        }

        return response()->json([
            'success' => true,
            'message' => 'Slots updated successfully',
            'slots' => $slots,
        ]);
    }

    /**
     * Delete a specific slot (Admin only).
     */
    public function deleteSlot(Request $request, $facilityId, $slotId): JsonResponse
    {
        $user = $request->user();

        $facility = Facility::where('society_id', $user->society_id)->find($facilityId);

        if (!$facility) {
            return response()->json([
                'success' => false,
                'message' => 'Facility not found',
            ], 404);
        }

        $slot = FacilitySlot::where('facility_id', $facilityId)->find($slotId);

        if (!$slot) {
            return response()->json([
                'success' => false,
                'message' => 'Slot not found',
            ], 404);
        }

        // Check if there are active bookings for this slot
        $hasActiveBookings = $slot->bookings()
            ->whereIn('booking_status', ['pending', 'confirmed'])
            ->where('booking_date', '>=', now()->toDateString())
            ->exists();

        if ($hasActiveBookings) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete slot with active bookings',
            ], 400);
        }

        $slot->delete();

        return response()->json([
            'success' => true,
            'message' => 'Slot deleted successfully',
        ]);
    }
}
