<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Facility;
use App\Models\FacilityBooking;
use App\Models\FacilitySlot;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use App\Services\PaymentGateway\SocietyPaymentGatewayFactory;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class FacilityBookingController extends Controller
{
    /**
     * Display a listing of bookings.
     */
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();

        $query = FacilityBooking::where('society_id', $user->society_id)
            ->with(['facility', 'slot', 'user', 'bookedBy']);

        // Role-based filtering
        if ($user->role->name === 'resident') {
            // Residents see only their own bookings
            $query->where('user_id', $user->id);
        }
        // Admin, Collector, Sub-Admin, Accountant can see all bookings

        // Filter by status
        if ($request->has('booking_status')) {
            $query->where('booking_status', $request->booking_status);
        }

        // Filter by payment status
        if ($request->has('payment_status')) {
            $query->where('payment_status', $request->payment_status);
        }

        // Filter by date range
        if ($request->has('from_date')) {
            $query->where('booking_date', '>=', $request->from_date);
        }
        if ($request->has('to_date')) {
            $query->where('booking_date', '<=', $request->to_date);
        }

        // Filter by facility
        if ($request->has('facility_id')) {
            $query->where('facility_id', $request->facility_id);
        }

        $bookings = $query->latest()->paginate($request->per_page ?? 15);

        return response()->json([
            'success' => true,
            'bookings' => $bookings,
        ]);
    }

    /**
     * Store a newly created booking.
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'facility_id' => 'required|exists:facilities,id',
            'slot_id' => 'required|exists:facility_slots,id',
            'booking_date' => 'required|date|after_or_equal:today',
            'user_id' => 'nullable|exists:users,id', // For collectors booking on behalf of residents
            'purpose' => 'nullable|string',
            'guest_count' => 'nullable|integer|min:1',
            'payment_method' => 'nullable|in:cash,online,cheque',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors(),
            ], 422);
        }

        $currentUser = $request->user();

        // Determine who the booking is for
        $userId = $request->user_id ?? $currentUser->id;

        // Check if collector/admin is booking for someone else
        $isBookingForOther = ($userId !== $currentUser->id);

        if ($isBookingForOther) {
            // Only collector, admin, sub-admin can book for others
            if (!in_array($currentUser->role->name, ['society_admin', 'collector', 'sub_admin'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'You do not have permission to book on behalf of others',
                ], 403);
            }
        }

        // Verify the user exists in the same society
        $bookingUser = User::where('id', $userId)
            ->where('society_id', $currentUser->society_id)
            ->first();

        if (!$bookingUser) {
            return response()->json([
                'success' => false,
                'message' => 'User not found in your society',
            ], 404);
        }

        // Get facility and verify it's active
        $facility = Facility::where('id', $request->facility_id)
            ->where('society_id', $currentUser->society_id)
            ->where('is_active', true)
            ->first();

        if (!$facility) {
            return response()->json([
                'success' => false,
                'message' => 'Facility not found or inactive',
            ], 404);
        }

        // Get slot and verify it's active
        $slot = FacilitySlot::where('id', $request->slot_id)
            ->where('facility_id', $facility->id)
            ->where('is_active', true)
            ->first();

        if (!$slot) {
            return response()->json([
                'success' => false,
                'message' => 'Slot not found or inactive',
            ], 404);
        }

        // Check availability
        if (!$facility->isSlotAvailable($slot->id, $request->booking_date)) {
            return response()->json([
                'success' => false,
                'message' => 'This slot is already booked for the selected date',
            ], 400);
        }

        // Create booking
        DB::beginTransaction();
        try {
            $booking = FacilityBooking::create([
                'society_id' => $currentUser->society_id,
                'facility_id' => $facility->id,
                'slot_id' => $slot->id,
                'user_id' => $userId,
                'booked_by' => $currentUser->id, // Track who made the booking
                'booking_date' => $request->booking_date,
                'amount' => $slot->price,
                'payment_status' => 'pending',
                'payment_method' => $request->payment_method,
                'booking_status' => ($facility->requires_approval || $request->payment_method === 'cash') ? 'pending' : 'confirmed',
                'purpose' => $request->purpose,
                'guest_count' => $request->guest_count,
            ]);

            DB::commit();

            // Load relationships
            $booking->load(['facility', 'slot', 'user', 'bookedBy']);

            return response()->json([
                'success' => true,
                'message' => ($facility->requires_approval || $request->payment_method === 'cash')
                    ? 'Booking request submitted successfully. Awaiting approval.'
                    : 'Booking confirmed successfully',
                'booking' => $booking,
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Failed to create booking',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Display the specified booking.
     */
    public function show(Request $request, $id): JsonResponse
    {
        $user = $request->user();

        $query = FacilityBooking::where('society_id', $user->society_id)
            ->with(['facility', 'slot', 'user', 'bookedBy']);

        // Residents can only view their own bookings
        if ($user->role->name === 'resident') {
            $query->where('user_id', $user->id);
        }

        $booking = $query->find($id);

        if (!$booking) {
            return response()->json([
                'success' => false,
                'message' => 'Booking not found',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'booking' => $booking,
        ]);
    }

    /**
     * Update booking status (Admin/Collector only).
     */
    public function updateStatus(Request $request, $id): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'booking_status' => 'required|in:pending,confirmed,cancelled,rejected,completed',
            'admin_notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        $booking = FacilityBooking::where('society_id', $user->society_id)->find($id);

        if (!$booking) {
            return response()->json([
                'success' => false,
                'message' => 'Booking not found',
            ], 404);
        }

        $booking->update([
            'booking_status' => $request->booking_status,
            'admin_notes' => $request->admin_notes,
        ]);

        $booking->load(['facility', 'slot', 'user', 'bookedBy']);

        return response()->json([
            'success' => true,
            'message' => 'Booking status updated successfully',
            'booking' => $booking,
        ]);
    }

    /**
     * Update payment status (Collector/Accountant only).
     */
    public function updatePayment(Request $request, $id): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'payment_status' => 'required|in:pending,paid,failed,refunded',
            'payment_method' => 'nullable|in:cash,online,cheque',
            'payment_details' => 'nullable|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        $booking = FacilityBooking::where('society_id', $user->society_id)->find($id);

        if (!$booking) {
            return response()->json([
                'success' => false,
                'message' => 'Booking not found',
            ], 404);
        }

        $updateData = [
            'payment_status' => $request->payment_status,
        ];

        if ($request->has('payment_method')) {
            $updateData['payment_method'] = $request->payment_method;
        }

        if ($request->has('payment_details')) {
            $updateData['payment_details'] = $request->payment_details;
        }

        $booking->update($updateData);

        $booking->load(['facility', 'slot', 'user', 'bookedBy']);

        return response()->json([
            'success' => true,
            'message' => 'Payment status updated successfully',
            'booking' => $booking,
        ]);
    }

    /**
     * Cancel a booking (Resident/Admin).
     */
    public function cancel(Request $request, $id): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'cancellation_reason' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = $request->user();

        $query = FacilityBooking::where('society_id', $user->society_id);

        // Residents can only cancel their own bookings
        if ($user->role->name === 'resident') {
            $query->where('user_id', $user->id);
        }

        $booking = $query->find($id);

        if (!$booking) {
            return response()->json([
                'success' => false,
                'message' => 'Booking not found',
            ], 404);
        }

        if (!$booking->canBeCancelled()) {
            return response()->json([
                'success' => false,
                'message' => 'This booking cannot be cancelled',
            ], 400);
        }

        $booking->update([
            'booking_status' => 'cancelled',
            'cancellation_reason' => $request->cancellation_reason,
        ]);

        $booking->load(['facility', 'slot', 'user', 'bookedBy']);

        return response()->json([
            'success' => true,
            'message' => 'Booking cancelled successfully',
            'booking' => $booking,
        ]);
    }

    /**
     * Check-in for a booking (Collector/Admin).
     */
    public function checkIn(Request $request, $id): JsonResponse
    {
        $user = $request->user();

        $booking = FacilityBooking::where('society_id', $user->society_id)
            ->where('booking_status', 'confirmed')
            ->find($id);

        if (!$booking) {
            return response()->json([
                'success' => false,
                'message' => 'Booking not found or not confirmed',
            ], 404);
        }

        if ($booking->checked_in_at) {
            return response()->json([
                'success' => false,
                'message' => 'Already checked in',
            ], 400);
        }

        $booking->update([
            'checked_in_at' => now(),
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Check-in recorded successfully',
            'booking' => $booking,
        ]);
    }

    /**
     * Check-out for a booking (Collector/Admin).
     */
    public function checkOut(Request $request, $id): JsonResponse
    {
        $user = $request->user();

        $booking = FacilityBooking::where('society_id', $user->society_id)
            ->where('booking_status', 'confirmed')
            ->find($id);

        if (!$booking) {
            return response()->json([
                'success' => false,
                'message' => 'Booking not found or not confirmed',
            ], 404);
        }

        if (!$booking->checked_in_at) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot check-out without check-in',
            ], 400);
        }

        if ($booking->checked_out_at) {
            return response()->json([
                'success' => false,
                'message' => 'Already checked out',
            ], 400);
        }

        $booking->update([
            'checked_out_at' => now(),
            'booking_status' => 'completed',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Check-out recorded successfully',
            'booking' => $booking,
        ]);
    }
    /**
     * Create Razorpay Order
     */
    public function createOrder(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'facility_id' => 'required|exists:facilities,id',
            'slot_id' => 'required|exists:facility_slots,id',
            'booking_date' => 'required|date',
            'amount' => 'required|numeric',
            'purpose' => 'nullable|string',
            'guest_count' => 'nullable|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Check availability before creating order
            $facility = Facility::find($request->facility_id);
            if (!$facility->isSlotAvailable($request->slot_id, $request->booking_date)) {
                return response()->json([
                    'success' => false,
                    'message' => 'This slot is already booked for the selected date',
                ], 400);
            }

            $user = $request->user();
            $user->loadMissing('society.settings');
            $societySettings = $user->society?->settings;

            if (!$societySettings || !$societySettings->razorpay_enabled) {
                return response()->json([
                    'success' => false,
                    'message' => 'Razorpay is not enabled for this society',
                ], 400);
            }

            $gateway = SocietyPaymentGatewayFactory::getGateway($societySettings, 'razorpay');
            if (!$gateway) {
                return response()->json([
                    'success' => false,
                    'message' => 'Razorpay credentials not configured for this society',
                ], 500);
            }

            // Create Razorpay Order (DB-driven keys)
            $orderData = $gateway->createOrder((float) $request->amount, 'INR', [
                'receipt' => 'facility_booking_' . time() . '_' . $user->id,
                'notes' => [
                    'type' => 'facility_booking',
                    'society_id' => (string) $user->society_id,
                    'facility_id' => (string) $request->facility_id,
                    'slot_id' => (string) $request->slot_id,
                    'booking_date' => (string) $request->booking_date,
                    'user_id' => (string) $user->id,
                ],
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Razorpay order created successfully',
                'data' => [
                    'order_id' => $orderData['order_id'],
                    'amount' => $orderData['amount'],
                    'currency' => $orderData['currency'] ?? 'INR',
                    'key_id' => $orderData['key'] ?? null,
                    'booking_data' => $request->all()
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating Razorpay order: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Verify Payment and Create Booking
     */
    public function verifyPayment(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'razorpay_order_id' => 'required',
            'razorpay_payment_id' => 'required',
            'razorpay_signature' => 'required',
            'facility_id' => 'required|exists:facilities,id',
            'slot_id' => 'required|exists:facility_slots,id',
            'booking_date' => 'required|date',
            'purpose' => 'nullable|string',
            'guest_count' => 'nullable|integer'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation errors',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $user = $request->user();
            $user->loadMissing('society.settings');
            $societySettings = $user->society?->settings;

            if (!$societySettings || !$societySettings->razorpay_enabled) {
                return response()->json([
                    'success' => false,
                    'message' => 'Razorpay is not enabled for this society',
                ], 400);
            }

            $gateway = SocietyPaymentGatewayFactory::getGateway($societySettings, 'razorpay');
            if (!$gateway) {
                return response()->json([
                    'success' => false,
                    'message' => 'Razorpay credentials not configured for this society',
                ], 500);
            }

            // Verify Signature (DB-driven secret)
            $result = $gateway->verifyPayment($request->all());
            if (!$result['success']) {
                return response()->json([
                    'success' => false,
                    'message' => $result['error'] ?? 'Payment verification failed: Invalid signature',
                ], 400);
            }

            // Fetch payment details (optional) for storing response
            $paymentDetails = $gateway->getPaymentDetails($request->razorpay_payment_id);

            // Create Booking
            $booking = FacilityBooking::create([
                'society_id' => $user->society_id, // Ensure society_id is set
                'facility_id' => $request->facility_id,
                'slot_id' => $request->slot_id,
                'user_id' => $user->id,
                'booked_by' => $user->id,
                'booking_date' => $request->booking_date,
                'amount' => $request->amount, // Trust server-side slot amount passed from UI
                'payment_status' => 'paid',
                'payment_method' => 'online',
                'booking_status' => 'confirmed',
                'purpose' => $request->purpose,
                'guest_count' => $request->guest_count,
                'razorpay_order_id' => $request->razorpay_order_id,
                'razorpay_payment_id' => $request->razorpay_payment_id,
                'razorpay_signature' => $request->razorpay_signature,
                'payment_response' => json_encode([
                    'verified' => true,
                    'gateway' => 'razorpay',
                    'payment' => $paymentDetails,
                    'raw' => $request->all(),
                ])
            ]);

            // Determine primary bank account for Revenue
            $primaryAccount = \App\Models\BankAccount::where('society_id', $user->society_id)
                ->where('is_primary', true)
                ->first();

            // Fallback to first account if no primary set
            if (!$primaryAccount) {
                $primaryAccount = \App\Models\BankAccount::where('society_id', $user->society_id)->first();
            }

            // Record Transaction
            if ($primaryAccount) {
                app(\App\Services\TransactionService::class)->recordIncome(
                    $booking->amount,
                    $primaryAccount->id,
                    'facility_booking',
                    $booking,
                    'online',
                    'Facility Booking: ' . $booking->facility->name,
                    $user->id
                );
            }

            $booking->load(['facility', 'slot', 'user']);

            return response()->json([
                'success' => true,
                'message' => 'Payment verified and booking created successfully',
                'data' => $booking
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating booking: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Handle Payment Failure
     */
    public function paymentFailed(Request $request): JsonResponse
    {
        // Log payment failure for monitoring and debugging
        \Illuminate\Support\Facades\Log::warning('Facility booking payment failed', [
            'user_id' => Auth::id(),
            'gateway' => 'razorpay',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Payment failure logged'
        ]);
    }
}
