<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\EmergencyAlert;
use App\Models\User;
use App\Services\FirebaseNotificationService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class EmergencyController extends Controller
{
    /**
     * Trigger an emergency panic alert.
     */
    public function trigger(Request $request, FirebaseNotificationService $firebaseService): JsonResponse
    {
        $user = $request->user();

        // 1. Create Emergency Alert
        $alert = EmergencyAlert::create([
            'society_id' => $user->society_id,
            'user_id' => $user->id,
            'unit_id' => $user->unit_id,
            'status' => 'triggered',
        ]);

        // 2. Prepare Notification Data
        $unitInfo = $user->unit ? $user->unit->unit_no : 'Unknown Unit';
        $userInfo = $user->name;
        
        // Try to get building info if available
        if ($user->unit && $user->unit->building) {
            $unitInfo .= ' (' . $user->unit->building->name . ')';
        }

        $body = "Panic Button pressed by {$userInfo} in {$unitInfo}!";

        // 3. Find Recipient Users based on user type
        // A. Guards and Admins - Always notified
        $adminRecipients = User::where('society_id', $user->society_id)
            ->whereHas('role', function ($q) {
                $q->whereIn('name', ['guard', 'society_admin', 'sub_admin', 'accountant']);
            })
            ->whereNotNull('fcm_token')
            ->get();

        $recipients = $adminRecipients;

        // B. Family Members - Only for Residents/Owners and Family Members (NOT for Tenants)
        $userType = $user->type; // 'owner', 'tenant', 'family_member', or null
        
        // If user is a tenant, only send to guards and admins (skip family members)
        if ($userType !== 'tenant') {
            // Identify "primary" user ID (owner of the household)
            // If sender is family member, use parent_id to find owner
            // If sender is owner (or null type), use own id
            $primaryUserId = null;
            if ($userType === 'family_member' && $user->parent_id) {
                $primaryUserId = $user->parent_id;
            } else {
                // User is owner or type is null (legacy owner)
                $primaryUserId = $user->id;
            }
            
            // Find all family members and owner of the same household
            // Include: owner + all family members with same parent_id + all users in same unit
            $familyRecipients = User::where('society_id', $user->society_id)
                ->where(function ($q) use ($primaryUserId, $user) {
                    // Include the owner (primary user) if not the sender
                    if ($primaryUserId != $user->id) {
                        $q->where('id', $primaryUserId);
                    }
                    // Include all family members with same parent_id
                    $q->orWhere(function ($subQ) use ($primaryUserId) {
                        $subQ->where('parent_id', $primaryUserId)
                             ->where('type', 'family_member');
                    });
                    // Also include all users in the same unit (as fallback)
                    if ($user->unit_id) {
                        $q->orWhere(function ($subQ) use ($user) {
                            $subQ->where('unit_id', $user->unit_id)
                                 ->where('id', '!=', $user->id)
                                 ->whereIn('type', ['owner', 'family_member', null]);
                        });
                    }
                })
                ->where('id', '!=', $user->id) // Exclude sender
                ->whereNotNull('fcm_token')
                ->get();

            $recipients = $recipients->merge($familyRecipients);
        }

        // 4. Send High Priority Notifications
        // We run this synchronously for now as it's critical. 
        // In a very large society, this should be queued, but for panic alerts, latency is key.
        $firebaseService->sendEmergencyNotification($recipients->all(), [
            'id' => $alert->id,
            'body' => $body,
        ]);

        // Prepare response message based on user type
        $message = 'Emergency alert triggered successfully. ';
        if ($userType === 'tenant') {
            $message .= 'Guards and administrators have been notified.';
        } else {
            $message .= 'Family members, guards, and administrators have been notified.';
        }

        return response()->json([
            'success' => true,
            'message' => $message,
            'alert_id' => $alert->id,
        ]);
    }

    /**
     * Check for active emergency alerts.
     */
    public function active(Request $request): JsonResponse
    {
        $user = $request->user();
        
        // Find latest active alert
        $alert = EmergencyAlert::where('society_id', $user->society_id)
            ->where('status', 'triggered') // Assuming 'triggered' is the active status
            ->with(['user', 'user.unit.building'])
            ->latest()
            ->first();

        if (!$alert) {
            return response()->json([
                'active' => false,
                'message' => 'No active emergency.',
            ]);
        }

        // Construct message details
        $residentName = $alert->user ? $alert->user->name : 'Unknown';
        $unitInfo = 'Unknown Unit';
        if ($alert->user && $alert->user->unit) {
            $unitInfo = $alert->user->unit->unit_no;
            if ($alert->user->unit->building) {
                $location = $alert->user->unit->building->name . ', ' . $unitInfo;
            } else {
                $location = $unitInfo;
            }
        } else {
            $location = 'Main Gate (Guard)'; // Or fallback
        }

        return response()->json([
            'active' => true,
            'message' => "Emergency declared by Resident ($residentName - $unitInfo)",
            'location' => $location,
            'timestamp' => $alert->created_at->toIso8601String(),
            'alert' => [
                'id' => $alert->id,
                'status' => $alert->status,
                'user' => [
                    'id' => $alert->user_id,
                    'name' => $residentName,
                    'phone' => $alert->user ? $alert->user->phone : null,
                ],
            ],
        ]);
    }

    /**
     * Resolve an emergency alert.
     */
    public function resolve(Request $request, $id): JsonResponse
    {
        $user = $request->user();

        // 1. Find Alert
        $alert = EmergencyAlert::where('id', $id)
            ->where('society_id', $user->society_id)
            ->first();

        if (!$alert) {
            return response()->json(['message' => 'Alert not found'], 404);
        }

        // 2. Check Permissions (Mobile Guard or Web Admin)
        if (!$user->hasRole('guard') && !$user->isWebAdmin()) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        if ($alert->status === 'resolved') {
            return response()->json(['message' => 'Alert is already resolved'], 200);
        }

        // 3. Update Status
        $alert->update([
            'status' => 'resolved',
            'resolved_at' => now(),
            'resolved_by' => $user->id,
            'notes' => $request->notes ?? null,
        ]);

        // 4. Send Resolution Notification
        try {
            $firebaseService = app(FirebaseNotificationService::class);
            
            // Get the original alert user
            $alertUser = User::find($alert->user_id);
            
            // Notify Admins, Guards, Original User, and their Family Members
            $recipients = User::where('society_id', $user->society_id)
                ->where(function ($q) use ($alert, $alertUser) {
                    // Admins and Guards
                    $q->whereHas('role', function ($r) {
                        $r->whereIn('name', ['guard', 'society_admin', 'sub_admin', 'accountant']);
                    })
                    // Original user
                    ->orWhere('id', $alert->user_id);
                    
                    // Family members of the original user
                    if ($alertUser) {
                        $primaryUserId = $alertUser->parent_id ?? $alertUser->id;
                        $q->orWhere(function ($familyQ) use ($primaryUserId, $alert) {
                            $familyQ->where('parent_id', $primaryUserId)
                                   ->where('type', 'family_member')
                                   ->where('id', '!=', $alert->user_id);
                        });
                        // Also include owner if alert was from family member
                        if ($alertUser->type === 'family_member' && $alertUser->parent_id) {
                            $q->orWhere('id', $alertUser->parent_id);
                        }
                    }
                })
                ->whereNotNull('fcm_token')
                ->get();

            $unitInfo = $alert->user && $alert->user->unit ? $alert->user->unit->unit_no : 'Unknown Unit';
            $resolverName = $user->name;
            
            $firebaseService->sendToUsers(
                $recipients->all(),
                'Emergency Resolved',
                "The emergency alert from {$unitInfo} has been resolved by {$resolverName}.",
                [
                    'type' => 'emergency_resolved',
                    'id' => (string) $alert->id,
                ]
            );
        } catch (\Exception $e) {
            // Log error for debugging notification failures (non-critical)
            \Illuminate\Support\Facades\Log::error('Failed to send resolution notification: ' . $e->getMessage());
        }

        return response()->json([
            'success' => true,
            'message' => 'Emergency alert marked as resolved.',
        ]);
    }
}
