<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Http\Resources\CommunityPostResource;
use App\Models\CommunityPost;
use Illuminate\Support\Facades\Auth;

class CommunityController extends Controller
{
    /**
     * List community posts.
     */
    /**
     * List community posts.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $societyId = $user->society_id;

        $posts = CommunityPost::where('society_id', $societyId)
            ->where('status', 'published')
            ->with('user')
            // Using withCount for real-time counts, but can fallback to columns if preferred
            ->withCount(['likes', 'comments'])
            ->withExists(['likes as is_liked' => function($q) use ($user) {
                $q->where('user_id', $user->id);
            }])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        return CommunityPostResource::collection($posts);
    }

    /**
     * Create a new community post.
     */
    public function store(Request $request)
    {
        $request->validate([
            'content' => 'required_without:media|nullable|string|max:1000',
            'media' => 'required_without:content|nullable|array',
            'media.*' => 'file|mimes:jpeg,png,jpg,gif,mp4,mov|max:20480', // 20MB
            'group_id' => 'nullable|exists:community_groups,id',
            'title' => 'nullable|string|max:255',
            'image' => 'nullable|string', // Backward compatibility
        ]);

        $user = Auth::user();

        // Use provided group or find/create a default "General" group
        $groupId = $request->group_id;
        if (! $groupId) {
            $group = \App\Models\CommunityGroup::firstOrCreate(
                ['society_id' => $user->society_id, 'name' => 'General'],
                ['description' => 'General community discussions', 'is_private' => false]
            );
            $groupId = $group->id;
        }

        $mediaUrls = [];
        $mediaType = 'text';

        // Handle Media Upload
        if ($request->hasFile('media')) {
            foreach ($request->file('media') as $file) {
                $path = $file->store('posts', 'public');
                $url = asset('storage/' . $path);
                $mediaUrls[] = $url;

                $mime = $file->getMimeType();
                if (str_starts_with($mime, 'video')) {
                    $mediaType = 'video';
                } else if ($mediaType !== 'video') {
                    $mediaType = 'image';
                }
            }
        } elseif ($request->image) {
            // Backward compatibility for base64/url string
            $mediaUrls[] = $request->image;
            $mediaType = 'image';
        }

        $post = CommunityPost::create([
            'society_id' => $user->society_id,
            'group_id' => $groupId,
            'user_id' => $user->id,
            'title' => $request->title ?? 'Post',
            'content' => $request->content,
            'image' => count($mediaUrls) > 0 ? $mediaUrls[0] : null, // Legacy column
            'media_type' => $mediaType,
            'media_urls' => $mediaUrls,
            'status' => 'published',
        ]);
        
        return new CommunityPostResource($post->load('user'));
    }

    /**
     * Toggles the like status of a community post.
     */
    public function like(Request $request, $id)
    {
        $post = CommunityPost::where('society_id', Auth::user()->society_id)
            ->findOrFail($id);
        
        $userId = Auth::id();
        
        $existingLike = \App\Models\PostLike::where('post_id', $post->id)
            ->where('user_id', $userId)
            ->first();

        if ($existingLike) {
            $existingLike->delete();
            $post->decrement('likes_count');
            $attached = false;
        } else {
            \App\Models\PostLike::create([
                'post_id' => $post->id,
                'user_id' => $userId
            ]);
            $post->increment('likes_count');
            $attached = true;
        }

        return response()->json([
            'attached' => $attached,
            'likes_count' => $post->fresh()->likes_count
        ]);
    }

    /**
     * Adds a comment to a community post.
     */
    public function comment(Request $request, $id)
    {
        $request->validate(['message' => 'required|string']);

        $post = CommunityPost::where('society_id', Auth::user()->society_id)
            ->findOrFail($id);

        $comment = \App\Models\PostComment::create([
            'post_id' => $post->id,
            'user_id' => Auth::id(),
            'message' => $request->message
        ]);
        
        $post->increment('comments_count');

        return response()->json([
            'data' => [
                'id' => $comment->id,
                'message' => $comment->message,
                'user' => [
                    'name' => Auth::user()->name,
                    'photo' => Auth::user()->profile_photo_url ?? null,
                ],
                'created_at' => $comment->created_at->diffForHumans()
            ]
        ]);
    }

    /**
     * Retrieves the resident directory with priority sorting.
     */
    public function directory(Request $request)
    {
        $user = Auth::user();
        
        $myUnitNumber = 0;
        if ($user->unit && preg_match('/(\d+)/', $user->unit->unit_no, $matches)) {
            $myUnitNumber = (int) $matches[1];
        }

        $residents = \App\Models\User::select('users.*')
            ->join('roles', 'users.role_id', '=', 'roles.id')
            ->leftJoin('units', 'users.unit_id', '=', 'units.id')
            ->where('users.society_id', $user->society_id)
            ->where('users.status', 'active')
            ->directoryVisible()
            ->selectRaw("
                CASE 
                    WHEN roles.name IN ('society_admin', 'admin') THEN 1
                    WHEN roles.name = 'sub_admin' THEN 2
                    WHEN roles.name = 'accountant' THEN 3
                    WHEN roles.name = 'collector' THEN 4
                    WHEN roles.name IN ('staff', 'guard') THEN 5
                    ELSE 6
                END as role_priority
            ")
            ->orderBy('role_priority', 'ASC')
            ->orderBy('users.name', 'ASC')
            ->with(['unit.building', 'role'])
            ->paginate(30);

        return response()->json($residents->map(function ($resident) {
            return [
                'id' => $resident->id,
                'name' => $resident->name,
                'photo' => $resident->profile_photo_url ? $resident->profile_photo_url : null,
                'unit' => $resident->unit ? ($resident->unit->building->name . ' - ' . $resident->unit->unit_no) : 'N/A',
                'role' => $resident->role->name,
                'role_priority' => $resident->role_priority, // Debug info
            ];
        }));
    }
}
