<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Comment;
use App\Models\MaintenanceTicket;
use App\Models\User;
use App\Services\FirebaseNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Role;
use App\Notifications\NewTicketComment;
use Illuminate\Support\Facades\Notification;

class CommentController extends Controller
{
    /**
     * List comments for a ticket.
     */
    public function index(Request $request, MaintenanceTicket $ticket)
    {
        // Authorization: Check if user can view this ticket
        $user = Auth::user();
        if ($user->id !== $ticket->user_id && !$user->isWebAdmin() && $user->role->name !== 'staff') {
             // Basic ownership check. Adjust if guards/vendors need access.
             // Vendors assigned to ticket should also see checks.
             if ($user->role->name === 'vendor' && $ticket->vendor_id !== $user->id) {
                 return response()->json(['message' => 'Unauthorized'], 403);
             }
             if ($user->role->name === 'resident' && $ticket->user_id !== $user->id) {
                 return response()->json(['message' => 'Unauthorized'], 403);
             }
        }

        $comments = $ticket->comments()
            ->with('user')
            ->orderBy('created_at', 'asc')
            ->get()
            ->map(function ($comment) {
                return [
                    'id' => $comment->id,
                    'user_id' => $comment->user_id,
                    'user_name' => $comment->user->name,
                    'user_role' => $comment->user->role->name,
                    'message' => $comment->message,
                    'images' => $comment->images,
                    'created_at' => $comment->created_at,
                    'is_me' => $comment->user_id === Auth::id(),
                ];
            });

        return response()->json(['data' => $comments]);
    }

    /**
     * Store a new comment.
     */
    public function store(Request $request, MaintenanceTicket $ticket, FirebaseNotificationService $firebaseService)
    {
        $request->validate([
            'message' => 'required|string',
            'images' => 'nullable|array',
        ]);

        $user = Auth::user();

        // Authorization (Simple check)
        if ($user->role->name === 'resident' && $ticket->user_id !== $user->id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Create Comment
        $comment = $ticket->comments()->create([
            'user_id' => $user->id,
            'message' => $request->message,
            'images' => $request->images,
            'is_internal' => false,
        ]);

        // Record Activity
        $ticket->recordTimeline('comment_added', "Comment added by {$user->name}");
        $ticket->touch();

        // Send Notifications
        // 1. Notify Resident (if comment is not from resident)
        if ($ticket->user_id !== $user->id && $ticket->user && $ticket->user->fcm_token) {
            $firebaseService->sendToUser(
                $ticket->user,
                'Ticket Update',
                "New comment on ticket #{$ticket->ticket_no}: " . \Illuminate\Support\Str::limit($comment->message, 50),
                [
                    'type' => 'ticket_comment',
                    'ticket_id' => (string) $ticket->id,
                ]
            );
        }

        // 2. Notify Assigned Vendor (if comment is not from vendor)
        if ($ticket->vendor_id && $ticket->vendor_id !== $user->id && $ticket->vendor && $ticket->vendor->fcm_token) {
            $firebaseService->sendToUser(
                $ticket->vendor,
                'Ticket Update',
                "New comment on ticket #{$ticket->ticket_no}: " . \Illuminate\Support\Str::limit($comment->message, 50),
                [
                    'type' => 'ticket_comment',
                    'ticket_id' => (string) $ticket->id,
                ]
            );
        }

        // 3. Notify Admins (if comment is not from an admin)
        // Or notify all admins except current user
        $admins = User::where('society_id', $ticket->society_id)
            ->whereHas('role', fn($q) => $q->whereIn('name', ['society_admin', 'sub_admin']))
            ->where('id', '!=', $user->id) 
            ->get();
            
        if ($admins->isNotEmpty()) {
            $firebaseService->sendToUsers(
                $admins->all(), 
                'New Ticket Comment', 
                "New comment on ticket #{$ticket->ticket_no} by {$user->name}", 
                [
                    'type' => 'ticket_comment',
                    'ticket_id' => (string) $ticket->id,
                ]
            );
        }

        return response()->json([
            'success' => true,
            'message' => 'Comment added successfully',
            'data' => $comment->load('user'),
        ]);
    }
}
