<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Bill;
use App\Models\Payment;
use App\Services\FirebaseNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Razorpay\Api\Api;

class CollectorPaymentController extends Controller
{
    /**
     * Fetch list of specific bills for collection.
     */
    public function bills(Request $request)
    {
        $user = Auth::guard('api')->user();

        if (!$user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $query = Bill::where('society_id', $user->society_id)
            ->with(['user', 'user.unit']);

        // Filter by Status
        if ($request->has('status')) {
            $status = $request->status;
            if ($status === 'unpaid') {
                $query->whereIn('status', ['due', 'partial']);
            } elseif ($status === 'overdue') {
                $query->where('status', 'overdue');
            } elseif ($status === 'paid') {
                $query->where('status', 'paid');
            }
        } else {
            // Default: Unpaid & Overdue
            $query->whereIn('status', ['due', 'partial', 'overdue']);
        }

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('bill_no', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($u) use ($search) {
                        $u->where('name', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%");
                    })
                    ->orWhereHas('user.unit', function ($un) use ($search) {
                        $un->where('unit_no', 'like', "%{$search}%");
                    });
            });
        }

        $bills = $query->orderBy('due_date', 'asc')->paginate(20);

        // Transform Data
        $data = $bills->getCollection()->map(function ($bill) {
            return [
                'id' => $bill->id,
                'bill_number' => $bill->bill_no,
                'unit_id' => $bill->user->unit_id ?? null,
                'unit_no' => $bill->user->unit->unit_no ?? 'N/A',
                'resident_name' => $bill->user->name ?? 'Unknown',
                'resident_phone' => $bill->user->phone ?? 'N/A',
                // Title isn't a direct column, constructing it or using description
                'title' => "Bill #{$bill->bill_no} - " . ($bill->billCycle ? $bill->billCycle->name : 'General'),
                'amount' => (float) $bill->amount,
                'due_date' => $bill->due_date->format('Y-m-d'),
                'status' => $bill->status,
                'is_overdue' => $bill->status === 'overdue',
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $data,
            'meta' => [
                'current_page' => $bills->currentPage(),
                'last_page' => $bills->lastPage(),
                'total' => $bills->total(),
            ]
        ]);
    }

    /**
     * Collect Payment (Cash or Online Request).
     */
    public function collect(Request $request, FirebaseNotificationService $firebaseService)
    {
        $request->validate([
            'bill_id' => 'required|exists:bills,id',
            'amount' => 'required|numeric|min:1',
            'method' => 'required|in:cash,upi_request',
            'notes' => 'nullable|string',
            'payer_vpa' => 'nullable|string',
        ]);

        $user = Auth::guard('api')->user();
        if (!$user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $bill = Bill::find($request->bill_id);
        
        // Authorization check
        if ($bill->society_id !== $user->society_id) {
            return response()->json(['message' => 'Unauthorized bill access'], 403);
        }

        // Logic for CASH
        if ($request->method === 'cash') {
            $payment = null;
            
            // Get primary bank account
            $primaryAccount = \App\Models\BankAccount::where('society_id', $user->society_id)
                ->where('is_primary', true)
                ->first();
            if (!$primaryAccount) {
                $primaryAccount = \App\Models\BankAccount::where('society_id', $user->society_id)->first();
            }
            
            DB::transaction(function () use ($request, $user, $bill, &$payment, $primaryAccount) {
                // Generate Payment No
                $paymentNo = 'PAY-' . strtoupper(substr($user->society->code ?? 'SOC', 0, 3)) . '-' . time();

                $payment = Payment::create([
                    'society_id' => $user->society_id,
                    'bill_id' => $bill->id,
                    'user_id' => $bill->user_id,
                    'payment_no' => $paymentNo,
                    'amount' => $request->amount,
                    'payment_method' => 'cash',
                    'status' => 'completed',
                    'bank_account_id' => $primaryAccount?->id,
                    'collected_by' => $user->id,
                    'collector_role' => 'collector',
                    'notes' => $request->notes,
                ]);

                // Credit collector's wallet for cash payments
                $wallet = \App\Models\CollectorWallet::getOrCreate($user->id, $user->society_id);
                $wallet->credit($payment->amount, $payment->id, 'Cash payment collected - Payment #' . $payment->payment_no);

                // Note: Transaction is NOT recorded here for cash payments
                // Cash goes to wallet first, then transaction is recorded when handover happens
                // This is different from online payments which go directly to bank account

                // Update Bill Status
                 $totalPaid = Payment::where('bill_id', $bill->id)->where('status', 'completed')->sum('amount');
                 
                 if (($totalPaid) >= $bill->amount) {
                     $bill->update(['status' => 'paid']);
                 } else {
                     $bill->update(['status' => 'partial']);
                 }
            });

            // Notify Resident
            if ($payment && $bill->user && $bill->user->fcm_token) {
                 $firebaseService->sendToUser(
                    $bill->user,
                    'Payment Received',
                    "Cash payment of ₹{$request->amount} received for Bill #{$bill->bill_no}",
                    ['type' => 'payment_received', 'payment_id' => (string) $payment->id]
                );
            }

            if (!$payment) {
                return response()->json(['message' => 'Failed to create payment'], 500);
            }

            return response()->json([
                'success' => true,
                'message' => 'Payment recorded successfully',
                'payment_status' => 'completed',
                'transaction_id' => $payment->payment_no // Local ID
            ]);
        }

        // Handle ONLINE payment method (UPI Request via Payment Link)
        if ($request->method === 'upi_request') {
            $residentVpa = $bill->user->upi_vpa ?? null;

            // Load society settings for payment gateway configuration
            $society = $user->society;
            $society->load('settings');
            $societySettings = $society->settings;

            // Validate Razorpay is enabled and configured
            if (!$societySettings || !$societySettings->razorpay_enabled) {
                Log::error('UPI Request: Razorpay not enabled', [
                    'society_id' => $society->id,
                ]);
                return response()->json(['message' => 'Razorpay is not enabled for this society. Please configure it in Society Settings.'], 400);
            }

            $keyId = $societySettings->razorpay_key_id;
            $keySecret = $societySettings->razorpay_key_secret;

            if (!$keyId || !$keySecret) {
                Log::error('UPI Request: Razorpay credentials missing', [
                    'society_id' => $society->id,
                ]);
                return response()->json(['message' => 'Payment gateway credentials not configured in Society Settings'], 500);
            }

            $api = new Api($keyId, $keySecret);
            $referenceId = 'PAYREQ_' . time() . '_' . $bill->id;

            // Sanitize and format phone number for Razorpay (E.164 format)
            $rawPhone = $bill->user->phone ?? '';
            $sanitizedPhone = preg_replace('/\D/', '', $rawPhone);
            
            if (strlen($sanitizedPhone) === 12 && str_starts_with($sanitizedPhone, '91')) {
                $sanitizedPhone = '+' . $sanitizedPhone;
            } elseif (strlen($sanitizedPhone) === 10) {
                $sanitizedPhone = '+91' . $sanitizedPhone;
            }
            
            try {
                $linkData = [
                    'amount' => $request->amount * 100,
                    'currency' => 'INR',
                    'accept_partial' => false,
                    'first_min_partial_amount' => 0,
                    'description' => "Payment for Bill #{$bill->bill_no}",
                    'customer' => [
                        'name' => $bill->user->name,
                        'contact' => $sanitizedPhone,
                        'email' => $bill->user->email ?? 'noreply@societify.com',
                    ],
                    'notify' => [
                        'sms' => true,
                        'email' => !empty($bill->user->email),
                    ],
                    'reminder_enable' => true,
                    'notes' => [
                        'bill_id' => (string) $bill->id,
                        'society_id' => (string) $user->society_id,
                        'collector_id' => (string) $user->id,
                    ],
                ];

                // Create Razorpay payment link
                $link = $api->paymentLink->create($linkData);

                // Create Pending Payment Record (with primary bank account for later transaction recording)
                $primaryAccount = \App\Models\BankAccount::where('society_id', $user->society_id)
                    ->where('is_primary', true)
                    ->first() ?? \App\Models\BankAccount::where('society_id', $user->society_id)->first();

                $payment = Payment::create([
                    'society_id' => $user->society_id,
                    'bill_id' => $bill->id,
                    'user_id' => $bill->user_id,
                    'payment_no' => $referenceId,
                    'amount' => $request->amount,
                    'payment_method' => 'upi_request',
                    'status' => 'pending',
                    'payment_request_id' => $link->id,
                    'bank_account_id' => $primaryAccount?->id,
                    'collected_by' => $user->id,
                    'collector_role' => 'collector',
                    'notes' => 'Payment Link Sent via Collector',
                ]);

                return response()->json([
                    'success' => true,
                    'message' => 'Payment request sent to resident',
                    'payment_status' => 'pending',
                    'request_id' => $link->id,
                    'payment_link' => $link->short_url,
                ]);

            } catch (\Exception $e) {
                // Log error for debugging payment gateway issues
                Log::error('UPI Request: Razorpay error', [
                    'society_id' => $society->id,
                    'bill_id' => $bill->id,
                    'error_message' => $e->getMessage(),
                    'error_code' => $e->getCode(),
                ]);
                return response()->json(['message' => 'Razorpay Error: ' . $e->getMessage()], 500);
            }
        }
    }

    /**
     * Check status of Payment Request.
     */
    public function checkStatus(Request $request, $requestId)
    {
         $user = Auth::guard('api')->user();
         // Basic Auth check
         if (!$user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized'], 403);
         }

         $payment = Payment::where('payment_request_id', $requestId)->first();
         if (!$payment) {
             return response()->json(['message' => 'Payment request not found in system'], 404);
         }

         // If already completed locally, return
         if ($payment->status === 'completed') {
             return response()->json([
                 'success' => true,
                 'request_id' => $requestId,
                 'status' => 'completed',
                 'amount_paid' => (float) $payment->amount,
                 'completed_at' => $payment->updated_at->toIso8601String(),
             ]);
         }

         // Poll Razorpay using Society-specific credentials
         $society = $user->society;
         $society->load('settings');
         $societySettings = $society->settings;

         if (!$societySettings || !$societySettings->razorpay_key_id || !$societySettings->razorpay_key_secret) {
             // Log warning when payment gateway credentials are missing
             Log::warning('checkStatus: Razorpay credentials missing for society ' . $society->id);
             return response()->json(['message' => 'Payment gateway credentials not configured'], 500);
         }

         $keyId = $societySettings->razorpay_key_id;
         $keySecret = $societySettings->razorpay_key_secret;
         $api = new Api($keyId, $keySecret);

         try {
             $link = $api->paymentLink->fetch($requestId);
             
             // Check status: 'paid', 'partially_paid', 'created', 'expired', 'cancelled'
             if ($link->status === 'paid') {
                 // Mark as completed
                 $payment->update([
                     'status' => 'completed',
                     'transaction_id' => $link->payments[0]['payment_id'] ?? 'RZP_LINK_PAID', // Get actual Pay ID if available
                     'reconciled_at' => now(), // Auto reconciled via API check
                 ]);

                 // Update Bill
                 $bill = $payment->bill;
                 $totalPaid = Payment::where('bill_id', $bill->id)->where('status', 'completed')->sum('amount');
                 if ($totalPaid >= $bill->amount) {
                     $bill->update(['status' => 'paid']);
                 } else {
                     $bill->update(['status' => 'partial']);
                 }

                 // [NEW] Record Transaction in Ledger
                 $primaryAccount = \App\Models\BankAccount::where('society_id', $user->society_id)
                    ->where('is_primary', true)
                    ->first();

                 // Fallback to first account
                 if (!$primaryAccount) {
                     $primaryAccount = \App\Models\BankAccount::where('society_id', $user->society_id)->first();
                 }

                 if ($primaryAccount) {
                     app(\App\Services\TransactionService::class)->recordIncome(
                         $payment->amount,
                         $primaryAccount->id,
                         'bill_payment',
                         $payment,
                         'online', // Online request is definitely online
                         'Online Payment Request #'.$requestId,
                         $user->id // Collected by (Collector who sent request) or System? Collector initiated.
                     );
                     
                     // Update payment with bank account
                     $payment->update(['bank_account_id' => $primaryAccount->id]);
                 }

                 return response()->json([
                    'success' => true,
                    'request_id' => $requestId,
                    'status' => 'completed',
                    'amount_paid' => (float) $payment->amount,
                    'completed_at' => now()->toIso8601String(),
                ]);
             } else {
                 // Return current RZP status
                 return response()->json([
                    'success' => true,
                    'request_id' => $requestId,
                    'status' => $link->status, // pending/created/etc
                    'amount_paid' => 0,
                ]);
             }

         } catch (\Exception $e) {
            return response()->json(['message' => 'Error checking status: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get collection history for the collector.
     */
    public function history(Request $request)
    {
        $user = Auth::guard('api')->user();
        
        if (!$user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $query = Payment::where('society_id', $user->society_id)
            ->where('collected_by', $user->id)
            ->where('status', 'completed')
            ->with(['user:id,name,phone', 'bill:id,bill_no', 'user.unit:id,unit_no']);

        // Filter by date range
        if ($request->has('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->has('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Filter by payment method
        if ($request->has('payment_method')) {
            $query->where('payment_method', $request->payment_method);
        }

        // Search
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('payment_no', 'like', "%{$search}%")
                  ->orWhereHas('user', function ($u) use ($search) {
                      $u->where('name', 'like', "%{$search}%")
                        ->orWhere('phone', 'like', "%{$search}%");
                  })
                  ->orWhereHas('bill', function ($b) use ($search) {
                      $b->where('bill_no', 'like', "%{$search}%");
                  });
            });
        }

        $payments = $query->orderBy('created_at', 'desc')
            ->paginate($request->get('per_page', 20));

        $data = $payments->getCollection()->map(function ($payment) {
            return [
                'id' => $payment->id,
                'payment_no' => $payment->payment_no,
                'amount' => (float) $payment->amount,
                'payment_method' => $payment->payment_method,
                'status' => $payment->status,
                'bill_no' => $payment->bill?->bill_no,
                'resident_name' => $payment->user?->name,
                'resident_phone' => $payment->user?->phone,
                'unit_no' => $payment->user?->unit?->unit_no,
                'notes' => $payment->notes,
                'created_at' => $payment->created_at->format('Y-m-d H:i:s'),
                'created_at_date' => $payment->created_at->format('Y-m-d'),
                'created_at_time' => $payment->created_at->format('H:i:s'),
            ];
        });

        // Calculate summary
        $summary = [
            'total_collected' => (float) Payment::where('society_id', $user->society_id)
                ->where('collected_by', $user->id)
                ->where('status', 'completed')
                ->when($request->has('date_from'), function ($q) use ($request) {
                    $q->whereDate('created_at', '>=', $request->date_from);
                })
                ->when($request->has('date_to'), function ($q) use ($request) {
                    $q->whereDate('created_at', '<=', $request->date_to);
                })
                ->sum('amount'),
            'total_count' => Payment::where('society_id', $user->society_id)
                ->where('collected_by', $user->id)
                ->where('status', 'completed')
                ->when($request->has('date_from'), function ($q) use ($request) {
                    $q->whereDate('created_at', '>=', $request->date_from);
                })
                ->when($request->has('date_to'), function ($q) use ($request) {
                    $q->whereDate('created_at', '<=', $request->date_to);
                })
                ->count(),
            'cash_collected' => (float) Payment::where('society_id', $user->society_id)
                ->where('collected_by', $user->id)
                ->where('status', 'completed')
                ->where('payment_method', 'cash')
                ->when($request->has('date_from'), function ($q) use ($request) {
                    $q->whereDate('created_at', '>=', $request->date_from);
                })
                ->when($request->has('date_to'), function ($q) use ($request) {
                    $q->whereDate('created_at', '<=', $request->date_to);
                })
                ->sum('amount'),
            'online_collected' => (float) Payment::where('society_id', $user->society_id)
                ->where('collected_by', $user->id)
                ->where('status', 'completed')
                ->whereIn('payment_method', ['online', 'upi_request'])
                ->when($request->has('date_from'), function ($q) use ($request) {
                    $q->whereDate('created_at', '>=', $request->date_from);
                })
                ->when($request->has('date_to'), function ($q) use ($request) {
                    $q->whereDate('created_at', '<=', $request->date_to);
                })
                ->sum('amount'),
        ];

        return response()->json([
            'success' => true,
            'data' => $data,
            'summary' => $summary,
            'meta' => [
                'current_page' => $payments->currentPage(),
                'last_page' => $payments->lastPage(),
                'total' => $payments->total(),
                'per_page' => $payments->perPage(),
            ]
        ]);
    }
}
