<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\Api\StoreCollectorPaymentRequest;
use App\Models\Bill;
use App\Models\CollectorWallet;
use App\Models\Payment;
use App\Models\Unit;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class CollectorController extends Controller
{
    /**
     * Get dashboard/summary data for collector.
     */
    public function dashboard(Request $request): JsonResponse
    {
        $user = Auth::guard('api')->user();

        if (! $user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized. Collector role required.'], 403);
        }

        $societyId = $user->society_id;
        $collectorId = $user->id;

        // Today's collections (only by this collector)
        $todayCollections = Payment::where('society_id', $societyId)
            ->where('collected_by', $collectorId)
            ->where('status', 'completed')
            ->whereDate('created_at', today())
            ->sum('amount');

        $todayCount = Payment::where('society_id', $societyId)
            ->where('collected_by', $collectorId)
            ->where('status', 'completed')
            ->whereDate('created_at', today())
            ->count();

        // This month's collections (only by this collector)
        $thisMonthCollections = Payment::where('society_id', $societyId)
            ->where('collected_by', $collectorId)
            ->where('status', 'completed')
            ->whereYear('created_at', now()->year)
            ->whereMonth('created_at', now()->month)
            ->sum('amount');

        $thisMonthCount = Payment::where('society_id', $societyId)
            ->where('collected_by', $collectorId)
            ->where('status', 'completed')
            ->whereYear('created_at', now()->year)
            ->whereMonth('created_at', now()->month)
            ->count();

        // Total outstanding (all due, overdue, partial bills)
        $totalOutstanding = Bill::where('society_id', $societyId)
            ->whereIn('status', ['due', 'overdue', 'partial'])
            ->sum('amount');

        // Overdue count (only overdue bills)
        $overdueCount = Bill::where('society_id', $societyId)
            ->where('status', 'overdue')
            ->count();

        // Due count (only due bills, not overdue)
        $dueCount = Bill::where('society_id', $societyId)
            ->where('status', 'due')
            ->count();

        // Total units
        $totalUnits = Unit::where('society_id', $societyId)->count();

        // Units with dues
        $dueUnits = Bill::where('society_id', $societyId)
            ->whereIn('status', ['due', 'overdue', 'partial'])
            ->distinct('user_id')
            ->count('user_id');

        // Cleared units (units with no outstanding bills)
        $clearedUnits = $totalUnits - $dueUnits;

        // Collection rate (based on this collector's collections)
        $totalBilled = Bill::where('society_id', $societyId)->sum('amount');
        $totalCollected = Payment::where('society_id', $societyId)
            ->where('collected_by', $collectorId)
            ->where('status', 'completed')
            ->sum('amount');
        $collectionRate = $totalBilled > 0 ? ($totalCollected / $totalBilled) * 100 : 0;

        // Get collector wallet balance
        $wallet = CollectorWallet::getOrCreate($user->id, $societyId);
        
        // Ensure wallet balance is synced with actual cash collections
        // Calculate actual cash collected and handover
        $actualCashCollected = Payment::where('society_id', $societyId)
            ->where('collected_by', $collectorId)
            ->where('status', 'completed')
            ->where('payment_method', 'cash')
            ->sum('amount');
        
        $actualCashHandover = $wallet->total_handover ?? 0;
        $expectedBalance = max(0, $actualCashCollected - $actualCashHandover);
        
        // Sync wallet if it's out of sync
        // If wallet total_collected is less than actual, it means some payments weren't credited
        $transactionCount = $wallet->transactions()->count();
        
        if ($wallet->total_collected < $actualCashCollected) {
            // Some payments weren't credited - sync the wallet
            $difference = $actualCashCollected - $wallet->total_collected;
            $wallet->balance += $difference; // Add the missing amount
            $wallet->total_collected = $actualCashCollected;
            $wallet->save();
            
            // Log the sync for audit
            \Illuminate\Support\Facades\Log::info('Collector wallet synced', [
                'collector_id' => $collectorId,
                'amount_added' => $difference,
                'new_balance' => $wallet->balance,
            ]);
        } elseif ($transactionCount == 0 && $expectedBalance != $wallet->balance) {
            // No transactions yet, but wallet is out of sync - sync it
            $wallet->balance = $expectedBalance;
            $wallet->total_collected = $actualCashCollected;
            $wallet->save();
        } elseif ($transactionCount > 0 && abs($expectedBalance - $wallet->balance) > 0.01) {
            // There are transactions, but balance is significantly off - log for investigation
            \Illuminate\Support\Facades\Log::warning('Collector wallet balance mismatch', [
                'collector_id' => $collectorId,
                'wallet_balance' => $wallet->balance,
                'expected_balance' => $expectedBalance,
                'actual_cash_collected' => $actualCashCollected,
                'total_handover' => $actualCashHandover,
            ]);
        }

        return response()->json([
            'today_collections' => (float) $todayCollections,
            'today_count' => $todayCount,
            'total_collected_today' => (float) $todayCollections, // For Flutter compatibility
            'this_month_collections' => (float) $thisMonthCollections,
            'this_month_count' => $thisMonthCount,
            'total_collected_month' => (float) $thisMonthCollections, // For Flutter compatibility
            'total_outstanding' => (float) $totalOutstanding,
            'total_due' => (float) $totalOutstanding, // Alias for backward compatibility
            'overdue_count' => $overdueCount,
            'due_count' => $dueCount,
            'total_units' => $totalUnits,
            'due_units' => $dueUnits,
            'cleared_units' => $clearedUnits,
            'collection_rate' => round($collectionRate, 2),
            'wallet' => [
                'balance' => (float) $wallet->fresh()->balance,
                'total_collected' => (float) $wallet->fresh()->total_collected,
                'total_handover' => (float) $wallet->fresh()->total_handover,
                'cash_in_hand' => (float) $wallet->fresh()->balance, // Alias for cash in hand
            ],
        ]);
    }

    /**
     * Get due list (all units with outstanding bills).
     */
    public function dues(Request $request): JsonResponse
    {
        $user = Auth::guard('api')->user();

        if (! $user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized. Collector role required.'], 403);
        }

        $societyId = $user->society_id;

        $query = Bill::where('society_id', $societyId)
            ->whereIn('status', ['due', 'overdue', 'partial'])
            ->with(['user:id,name,phone', 'unit:id,unit_no,building_id', 'unit.building:id,name']);

        // Filter by building
        if ($request->has('building_id')) {
            $query->whereHas('unit', function ($q) use ($request) {
                $q->where('building_id', $request->get('building_id'));
            });
        }

        // Filter by status
        if ($request->has('status') && $request->get('status') !== 'all') {
            $query->where('status', $request->get('status'));
        }

        // Search
        if ($request->has('search')) {
            $search = $request->get('search');
            $query->where(function ($q) use ($search) {
                $q->where('bill_no', 'like', "%{$search}%")
                    ->orWhereHas('unit', function ($q) use ($search) {
                        $q->where('unit_no', 'like', "%{$search}%");
                    })
                    ->orWhereHas('user', function ($q) use ($search) {
                        $q->where('name', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%");
                    });
            });
        }

        $bills = $query->get();

        // Group by unit and calculate totals
        $unitsData = $bills->groupBy('unit_id')->map(function ($unitBills, $unitId) {
            $firstBill = $unitBills->first();
            $unit = $firstBill->unit;
            $user = $firstBill->user;

            $totalDue = $unitBills->sum('amount');
            $overdueAmount = $unitBills->where('status', 'overdue')->sum('amount');

            return [
                'unit_id' => $unitId,
                'unit_no' => $unit ? $unit->unit_no : null,
                'building_name' => $unit && $unit->building ? $unit->building->name : null,
                'resident_name' => $user ? $user->name : null,
                'resident_phone' => $user ? $user->phone : null,
                'total_due' => (float) $totalDue,
                'overdue_amount' => (float) $overdueAmount,
                'bills' => $unitBills->map(function ($bill) {
                    return [
                        'id' => $bill->id,
                        'bill_no' => $bill->bill_no,
                        'amount' => (float) $bill->amount,
                        'due_date' => $bill->due_date->format('Y-m-d'),
                        'status' => $bill->status,
                        'days_overdue' => $bill->status === 'overdue' ? now()->diffInDays($bill->due_date) : 0,
                    ];
                })->values(),
            ];
        })->values();

        return response()->json([
            'data' => $unitsData,
            'total_units' => $unitsData->count(),
            'total_amount' => (float) $unitsData->sum('total_due'),
        ]);
    }

    /**
     * Get cleared list (units with no outstanding bills).
     */
    public function cleared(Request $request): JsonResponse
    {
        $user = Auth::guard('api')->user();

        if (! $user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized. Collector role required.'], 403);
        }

        $societyId = $user->society_id;

        $query = Unit::where('society_id', $societyId)
            ->whereHas('user')
            ->with(['user:id,name,phone', 'building:id,name']);

        // Filter by building
        if ($request->has('building_id')) {
            $query->where('building_id', $request->get('building_id'));
        }

        // Get units with no outstanding bills
        $units = $query->get()->filter(function ($unit) use ($societyId) {
            $outstandingBills = Bill::where('society_id', $societyId)
                ->where('user_id', $unit->user_id)
                ->whereIn('status', ['due', 'overdue', 'partial'])
                ->exists();

            return ! $outstandingBills;
        });

        // Get payment data for cleared units
        $unitsData = $units->map(function ($unit) use ($societyId, $request) {
            $payments = Payment::where('society_id', $societyId)
                ->where('user_id', $unit->user_id)
                ->where('status', 'completed');

            // Filter by date range if provided
            if ($request->has('date_from')) {
                $payments->whereDate('created_at', '>=', $request->get('date_from'));
            }
            if ($request->has('date_to')) {
                $payments->whereDate('created_at', '<=', $request->get('date_to'));
            }

            $paymentData = $payments->get();

            return [
                'unit_id' => $unit->id,
                'unit_no' => $unit->unit_no,
                'building_name' => $unit->building ? $unit->building->name : null,
                'resident_name' => $unit->user ? $unit->user->name : null,
                'total_paid' => (float) $paymentData->sum('amount'),
                'last_payment_date' => $paymentData->isNotEmpty() ? $paymentData->max('created_at')->format('Y-m-d') : null,
                'payment_count' => $paymentData->count(),
            ];
        })->values();

        return response()->json([
            'data' => $unitsData,
            'total_units' => $unitsData->count(),
        ]);
    }

    /**
     * Record a payment.
     */
    public function storePayment(StoreCollectorPaymentRequest $request): JsonResponse
    {
        $user = Auth::guard('api')->user();

        if (! $user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized. Collector role required.'], 403);
        }

        $societyId = $user->society_id;
        $data = $request->validated();
        $data['society_id'] = $societyId;
        $data['created_by'] = $user->id;
        $data['status'] = 'completed'; // Collector payments are always completed

        // Get user_id from unit_id
        $unit = Unit::find($data['unit_id']);
        if ($unit && $unit->user_id) {
            $data['user_id'] = $unit->user_id;
        }

        // Generate payment number if not provided
        if (empty($data['payment_no'])) {
            $society = $user->society;
            $paymentNoPrefix = 'PAY-'.strtoupper(substr($society->code ?? 'SOC', 0, 3)).'-'.date('Y').'-';
            $paymentNo = $paymentNoPrefix.str_pad(time(), 8, '0', STR_PAD_LEFT);
            $data['payment_no'] = $paymentNo;
        }

        $payment = null;

        DB::transaction(function () use ($data, $societyId, &$payment, $user) {
            // Add Collector details
            $data['collected_by'] = $user->id;
            $data['collector_role'] = 'collector';
            // Determine Bank Account (Primary or Fallback)
            $primaryAccount = \App\Models\BankAccount::where('society_id', $societyId)->where('is_primary', true)->first();
            
            // Fallback
            if (!$primaryAccount) {
                $primaryAccount = \App\Models\BankAccount::where('society_id', $societyId)->first();
            }

            $data['bank_account_id'] = $primaryAccount ? $primaryAccount->id : null;

            $payment = Payment::create($data);

            // Credit collector's wallet for CASH payments
            if ($data['payment_method'] === 'cash') {
                $wallet = CollectorWallet::getOrCreate($user->id, $societyId);
                $wallet->credit($payment->amount, $payment->id, 'Payment #' . $payment->payment_no);
            }

            // Record Transaction (only for non-cash - cash goes to wallet first)
            if ($payment->bank_account_id && $data['payment_method'] !== 'cash') {
                app(\App\Services\TransactionService::class)->recordIncome(
                    $payment->amount,
                    $payment->bank_account_id,
                    'bill_payment',
                    $payment,
                    $payment->payment_method,
                    'Payment collected by ' . $user->name,
                    $user->id
                );
            }

            // Update bill statuses if bills are associated
            if (! empty($data['bill_ids']) && is_array($data['bill_ids'])) {
                foreach ($data['bill_ids'] as $billId) {
                    $bill = Bill::find($billId);

                    if ($bill && $bill->society_id === $societyId) {
                        // Calculate total payments for this bill
                        $totalPayments = Payment::where('bill_id', $bill->id)
                            ->where('status', 'completed')
                            ->sum('amount');

                        // Update bill status
                        if ($totalPayments >= $bill->amount) {
                            $bill->update(['status' => 'paid']);
                        } elseif ($totalPayments > 0) {
                            $bill->update(['status' => 'partial']);
                        }
                    }
                }
            } elseif (! empty($data['bill_id'])) {
                // Single bill association (backward compatibility)
                $bill = Bill::find($data['bill_id']);

                if ($bill && $bill->society_id === $societyId) {
                    $totalPayments = Payment::where('bill_id', $bill->id)
                        ->where('status', 'completed')
                        ->sum('amount');

                    if ($totalPayments >= $bill->amount) {
                        $bill->update(['status' => 'paid']);
                    } elseif ($totalPayments > 0) {
                        $bill->update(['status' => 'partial']);
                    }
                }
            }
        });

        if (!$payment) {
            return response()->json(['message' => 'Failed to create payment'], 500);
        }

        /** @var Payment $payment */
        $paymentRecord = $payment;
        $paymentRecord->load(['user:id,name,phone', 'bill:id,bill_no']);

        // Send Notification to Resident
        try {
            if ($paymentRecord->user && $paymentRecord->user->fcm_token) {
                $firebaseService = app(\App\Services\FirebaseNotificationService::class);
                $title = 'Payment Received';
                $body = "Payment of ₹{$paymentRecord->amount} received via " . strtoupper($paymentRecord->payment_method);
                
                $firebaseService->sendToUser($paymentRecord->user, $title, $body, [
                    'type' => 'payment_received',
                    'payment_id' => (string) $paymentRecord->id,
                    'amount' => (string) $paymentRecord->amount,
                ]);
            }
        } catch (\Exception $e) {
            // Log warning for debugging notification failures (non-critical)
            \Illuminate\Support\Facades\Log::warning('Failed to send payment notification: ' . $e->getMessage());
        }

        return response()->json([
            'message' => 'Payment recorded successfully.',
            'payment' => [
                'id' => $paymentRecord->id,
                'payment_no' => $paymentRecord->payment_no,
                'amount' => (float) $paymentRecord->amount,
                'payment_method' => $paymentRecord->payment_method,
                'status' => $paymentRecord->status,
                'created_at' => $paymentRecord->created_at->format('Y-m-d H:i:s'),
            ],
        ], 201);
    }

    /**
     * Handover collected cash to the bank/admin.
     */
    public function handover(Request $request): JsonResponse
    {
        $user = Auth::guard('api')->user();

        if (! $user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized. Collector role required.'], 403);
        }

        $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'notes' => 'nullable|string',
        ]);

        $societyId = $user->society_id;
        $amount = $request->get('amount');
        $notes = $request->get('notes');

        // Check wallet balance
        $wallet = CollectorWallet::getOrCreate($user->id, $societyId);
        if ($amount > $wallet->balance) {
            return response()->json([
                'message' => 'Insufficient wallet balance. You have ₹' . number_format($wallet->balance, 2) . ' in your wallet.',
            ], 400);
        }

        // Find primary bank account or fallback
        $primaryAccount = \App\Models\BankAccount::where('society_id', $societyId)->where('is_primary', true)->first();
        if (! $primaryAccount) {
            $primaryAccount = \App\Models\BankAccount::where('society_id', $societyId)->first();
        }

        if (! $primaryAccount) {
            return response()->json(['message' => 'No bank account found. Please create one first.'], 500);
        }

        try {
            DB::transaction(function () use ($wallet, $amount, $notes, $user, $primaryAccount) {
                // Debit the collector's wallet
                $wallet->debit($amount, $user->id, $notes ?? 'Cash handed over');

                // Record bank transaction
                app(\App\Services\TransactionService::class)->handleCashHandover(
                    $user->id,
                    $primaryAccount->id,
                    $amount,
                    $notes
                );
            });

            return response()->json([
                'message' => 'Cash handed over successfully.',
                'wallet' => [
                    'balance' => (float) $wallet->fresh()->balance,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Handover failed: ' . $e->getMessage()], 500);
        }
    }

    /**
     * Get collection report.
     */
    public function reports(Request $request): JsonResponse
    {
        $user = Auth::guard('api')->user();

        if (! $user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized. Collector role required.'], 403);
        }

        $societyId = $user->society_id;

        // Date range is required
        $dateFrom = $request->get('date_from', now()->startOfMonth()->format('Y-m-d'));
        $dateTo = $request->get('date_to', now()->format('Y-m-d'));

        $query = Payment::where('society_id', $societyId)
            ->where('status', 'completed')
            ->whereBetween('created_at', [$dateFrom, $dateTo.' 23:59:59']);

        // Filter by building
        if ($request->has('building_id')) {
            $query->whereHas('user.unit', function ($q) use ($request) {
                $q->where('building_id', $request->get('building_id'));
            });
        }

        $payments = $query->get();

        // Summary
        $totalCollected = $payments->sum('amount');
        $totalTransactions = $payments->count();

        // By payment method
        $byMethod = $payments->groupBy('payment_method')->map(function ($methodPayments, $method) {
            return [
                'method' => $method,
                'amount' => (float) $methodPayments->sum('amount'),
                'count' => $methodPayments->count(),
            ];
        })->values();

        // By collector (if multiple collectors)
        $byCollector = $payments->whereNotNull('created_by')
            ->groupBy('created_by')
            ->map(function ($collectorPayments, $collectorId) {
                $collector = \App\Models\User::find($collectorId);

                return [
                    'collector_id' => $collectorId,
                    'collector_name' => $collector ? $collector->name : 'Unknown',
                    'amount' => (float) $collectorPayments->sum('amount'),
                    'count' => $collectorPayments->count(),
                ];
            })
            ->values();

        // Daily breakdown
        $dailyBreakdown = $payments->groupBy(function ($payment) {
            return $payment->created_at->format('Y-m-d');
        })->map(function ($dayPayments, $date) {
            return [
                'date' => $date,
                'amount' => (float) $dayPayments->sum('amount'),
                'count' => $dayPayments->count(),
            ];
        })->sortBy('date')->values();

        return response()->json([
            'period' => [
                'from' => $dateFrom,
                'to' => $dateTo,
            ],
            'summary' => [
                'total_collected' => (float) $totalCollected,
                'total_transactions' => $totalTransactions,
                'cash' => (float) $payments->where('payment_method', 'cash')->sum('amount'),
                'cheque' => (float) $payments->where('payment_method', 'cheque')->sum('amount'),
                'online' => (float) $payments->where('payment_method', 'online')->sum('amount'),
            ],
            'by_method' => $byMethod,
            'by_collector' => $byCollector,
            'daily_breakdown' => $dailyBreakdown,
        ]);
    }

    /**
     * Get unit details with bills and payment history.
     */
    public function unitDetails(Request $request, int $unitId): JsonResponse
    {
        $user = Auth::guard('api')->user();

        if (! $user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized. Collector role required.'], 403);
        }

        $societyId = $user->society_id;

        $unit = Unit::where('society_id', $societyId)
            ->with(['building:id,name', 'user:id,name,phone'])
            ->find($unitId);

        if (! $unit) {
            return response()->json(['message' => 'Unit not found.'], 404);
        }

        // Get bills for the unit's resident
        $bills = Bill::where('society_id', $societyId)
            ->where('user_id', $unit->user_id)
            ->orderBy('due_date', 'asc')
            ->get()
            ->map(function ($bill) {
                return [
                    'id' => $bill->id,
                    'bill_no' => $bill->bill_no,
                    'amount' => (float) $bill->amount,
                    'due_date' => $bill->due_date->format('Y-m-d'),
                    'status' => $bill->status,
                    'days_overdue' => $bill->status === 'overdue' ? now()->diffInDays($bill->due_date) : 0,
                ];
            });

        // Calculate total due
        $totalDue = Bill::where('society_id', $societyId)
            ->where('user_id', $unit->user_id)
            ->whereIn('status', ['due', 'overdue', 'partial'])
            ->sum('amount');

        // Payment history
        $paymentHistory = Payment::where('society_id', $societyId)
            ->where('user_id', $unit->user_id)
            ->where('status', 'completed')
            ->orderBy('created_at', 'desc')
            ->limit(20)
            ->get()
            ->map(function ($payment) {
                return [
                    'date' => $payment->created_at->format('Y-m-d'),
                    'amount' => (float) $payment->amount,
                    'method' => $payment->payment_method,
                    'payment_no' => $payment->payment_no,
                ];
            });

        return response()->json([
            'unit' => [
                'id' => $unit->id,
                'unit_no' => $unit->unit_no,
                'building_name' => $unit->building ? $unit->building->name : null,
                'resident' => $unit->user ? [
                    'name' => $unit->user->name,
                    'phone' => $unit->user->phone,
                ] : null,
            ],
            'bills' => $bills,
            'total_due' => (float) $totalDue,
            'payment_history' => $paymentHistory,
        ]);
    }

    /**
     * Get collector wallet details with transaction history.
     */
    public function wallet(Request $request): JsonResponse
    {
        $user = Auth::guard('api')->user();

        if (! $user || $user->role->name !== 'collector') {
            return response()->json(['message' => 'Unauthorized. Collector role required.'], 403);
        }

        $societyId = $user->society_id;
        $wallet = CollectorWallet::getOrCreate($user->id, $societyId);

        // Get recent transactions
        $transactions = $wallet->transactions()
            ->with('receiver:id,name')
            ->orderBy('created_at', 'desc')
            ->limit(50)
            ->get()
            ->map(function ($txn) {
                return [
                    'id' => $txn->id,
                    'type' => $txn->type,
                    'amount' => (float) $txn->amount,
                    'balance_after' => (float) $txn->balance_after,
                    'reference_type' => $txn->reference_type,
                    'notes' => $txn->notes,
                    'received_by' => $txn->receiver?->name,
                    'created_at' => $txn->created_at->format('Y-m-d H:i:s'),
                ];
            });

        return response()->json([
            'wallet' => [
                'balance' => (float) $wallet->balance,
                'total_collected' => (float) $wallet->total_collected,
                'total_handover' => (float) $wallet->total_handover,
            ],
            'transactions' => $transactions,
        ]);
    }
}
