<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    /**
     * Login with phone and password.
     */
    public function login(Request $request)
    {
        $request->validate([
            'phone' => 'required|string',
            'password' => 'required|string',
        ]);

        $user = User::where('phone', $request->phone)->first();

        if (! $user || ! Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages([
                'phone' => ['The provided credentials are incorrect.'],
            ]);
        }

        // Check for Tenant Expiry
        if ($user->type === 'tenant' && $user->tenant_agreement_end_date) {
            $endDate = \Carbon\Carbon::parse($user->tenant_agreement_end_date);
            if ($endDate->isPast()) {
                throw ValidationException::withMessages([
                    'phone' => ['Your rent agreement has expired. Please contact the owner for renewal.'],
                ]);
            }
        }

        // Create token
        $token = $user->createToken('mobile-app')->plainTextToken;

        return response()->json([
            'token' => $token,
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'phone' => $user->phone,
                'email' => $user->email,
                'role' => $user->role->name ?? 'resident',
                'society_id' => $user->society_id,
                'type' => $user->type,
                'unit_id' => $user->unit_id,
                'parent_id' => $user->parent_id,
            ],
        ]);
    }

    /**
     * Send OTP for phone verification.
     */
    public function sendOtp(Request $request)
    {
        $request->validate([
            'phone' => 'required|string|regex:/^[0-9]{10}$/',
        ]);

        // Generate 6-digit OTP
        $otp = rand(100000, 999999);

        // Store OTP in cache for 5 minutes
        cache()->put("otp_{$request->phone}", $otp, now()->addMinutes(5));

        // In production, send via SMS service (Twilio, MSG91, etc.)
        // Never return OTPs in API responses.
        // For local/testing, log only a minimal hint for developer convenience.
        if (app()->environment(['local', 'testing'])) {
            Log::info('OTP generated (local/testing)', [
                'phone' => $request->phone,
                // Avoid logging full OTP even in dev; log only a minimal hint.
                'otp_last2' => substr((string) $otp, -2),
            ]);
        }

        return response()->json([
            'message' => 'OTP sent successfully',
        ]);
    }

    /**
     * Verify OTP and register/login user.
     */
    public function verifyOtp(Request $request)
    {
        $request->validate([
            'phone' => 'required|string',
            'otp' => 'required|string',
            'name' => 'nullable|string',
            'email' => 'nullable|email',
        ]);

        $cachedOtp = cache()->get("otp_{$request->phone}");

        if (! $cachedOtp || $cachedOtp != $request->otp) {
            throw ValidationException::withMessages([
                'otp' => ['Invalid or expired OTP.'],
            ]);
        }

        // Clear OTP
        cache()->forget("otp_{$request->phone}");

        // Find or create user
        $user = User::where('phone', $request->phone)->first();

        if (! $user) {
            // New user registration
            $user = User::create([
                'phone' => $request->phone,
                'name' => $request->name ?? 'User',
                'email' => $request->email,
                'password' => Hash::make(uniqid()), // Random password for OTP-based users
                'role_id' => \App\Models\Role::where('name', 'resident')->first()->id,
            ]);
        }

        // Check for Tenant Expiry (if existing user)
        if ($user->type === 'tenant' && $user->tenant_agreement_end_date) {
            $endDate = \Carbon\Carbon::parse($user->tenant_agreement_end_date);
            if ($endDate->isPast()) {
                throw ValidationException::withMessages([
                    'otp' => ['Your rent agreement has expired. Please contact the owner for renewal.'],
                ]);
            }
        }

        // Create token
        $token = $user->createToken('mobile-app')->plainTextToken;

        return response()->json([
            'token' => $token,
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'phone' => $user->phone,
                'email' => $user->email,
                'role' => $user->role->name ?? 'resident',
                'society_id' => $user->society_id,
            ],
        ]);
    }

    /**
     * Logout user.
     */
    public function logout(Request $request)
    {
        $request->user()->currentAccessToken()->delete();

        return response()->json([
            'message' => 'Logged out successfully',
        ]);
    }
}
