<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Http\Resources\AnnouncementResource;
use App\Models\Announcement;
use App\Models\AnnouncementAcknowledgement;
use App\Models\AnnouncementView;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class AnnouncementController extends Controller
{
    /**
     * List announcements relevant to the user.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $societyId = $user->society_id;

        // Load user relationships needed for filtering
        $user->load(['role', 'unit']);

        $query = Announcement::where('society_id', $societyId)
            ->where('is_published', true)
            // Show all published announcements that haven't expired
            // start_date is not used for filtering - if it's published, it's visible
            // Only filter by end_date to hide expired announcements
            ->where(function ($q) {
                $q->whereNull('end_date')
                    ->orWhere('end_date', '>=', now());
            });

        // Get all announcements first, then filter in PHP for better compatibility
        $allAnnouncements = $query->orderBy('priority', 'desc')
            ->orderBy('created_at', 'desc')
            ->get();

        // Filter announcements based on targeting
        $filteredAnnouncements = $allAnnouncements->filter(function ($announcement) use ($user) {
            // Get target arrays (they are already cast to arrays by the model)
            $targetRoles = $announcement->target_roles ?? null;
            $targetUnits = $announcement->target_units ?? null;
            $targetBuildings = $announcement->target_buildings ?? null;
            
            // Normalize to arrays (handle null, empty string, or array)
            if ($targetRoles === null || $targetRoles === '' || $targetRoles === '[]') {
                $targetRoles = [];
            } elseif (!is_array($targetRoles)) {
                $targetRoles = is_string($targetRoles) ? json_decode($targetRoles, true) : [];
                if (!is_array($targetRoles)) {
                    $targetRoles = [];
                }
            }
            
            if ($targetUnits === null || $targetUnits === '' || $targetUnits === '[]') {
                $targetUnits = [];
            } elseif (!is_array($targetUnits)) {
                $targetUnits = is_string($targetUnits) ? json_decode($targetUnits, true) : [];
                if (!is_array($targetUnits)) {
                    $targetUnits = [];
                }
            }
            
            if ($targetBuildings === null || $targetBuildings === '' || $targetBuildings === '[]') {
                $targetBuildings = [];
            } elseif (!is_array($targetBuildings)) {
                $targetBuildings = is_string($targetBuildings) ? json_decode($targetBuildings, true) : [];
                if (!is_array($targetBuildings)) {
                    $targetBuildings = [];
                }
            }
            
            // Remove null, empty string, or invalid values from arrays
            $targetRoles = array_filter($targetRoles, fn($v) => $v !== null && $v !== '' && $v !== 'null');
            $targetUnits = array_filter($targetUnits, fn($v) => $v !== null && $v !== '' && $v !== 'null');
            $targetBuildings = array_filter($targetBuildings, fn($v) => $v !== null && $v !== '' && $v !== 'null');
            
            // If all targets are empty/null, show to all
            $hasNoTargets = empty($targetRoles) && empty($targetUnits) && empty($targetBuildings);
            
            if ($hasNoTargets) {
                return true;
            }
            
            // Check if user matches any target
            $roleMatches = false;
            if (!empty($targetRoles) && $user->role) {
                // Check if role name matches
                $roleMatches = in_array($user->role->name, $targetRoles, true);
            }
            
            $unitMatches = false;
            if (!empty($targetUnits) && $user->unit_id) {
                // Check if unit ID matches (handle both string and int)
                $unitMatches = in_array($user->unit_id, array_map('intval', $targetUnits), true) 
                            || in_array((string)$user->unit_id, array_map('strval', $targetUnits), true);
            }
            
            $buildingMatches = false;
            if (!empty($targetBuildings) && $user->unit && $user->unit->building_id) {
                // Check if building ID matches (handle both string and int)
                $buildingMatches = in_array($user->unit->building_id, array_map('intval', $targetBuildings), true)
                                || in_array((string)$user->unit->building_id, array_map('strval', $targetBuildings), true);
            }
            
            $shouldShow = $roleMatches || $unitMatches || $buildingMatches;
            
            // Log for debugging (remove in production if needed)
            if (config('app.debug')) {
                Log::debug('Announcement filtering', [
                    'announcement_id' => $announcement->id,
                    'user_id' => $user->id,
                    'user_role' => $user->role?->name,
                    'user_unit_id' => $user->unit_id,
                    'user_building_id' => $user->unit?->building_id,
                    'target_roles' => $targetRoles,
                    'target_units' => $targetUnits,
                    'target_buildings' => $targetBuildings,
                    'has_no_targets' => $hasNoTargets,
                    'role_matches' => $roleMatches,
                    'unit_matches' => $unitMatches,
                    'building_matches' => $buildingMatches,
                    'should_show' => $shouldShow,
                ]);
            }
            
            return $shouldShow;
        });

        // Log summary for debugging
        if (config('app.debug')) {
            Log::debug('Announcement filtering summary', [
                'total_announcements' => $allAnnouncements->count(),
                'filtered_announcements' => $filteredAnnouncements->count(),
                'user_id' => $user->id,
                'user_role' => $user->role?->name,
                'user_unit_id' => $user->unit_id,
                'user_building_id' => $user->unit?->building_id,
                'society_id' => $societyId,
            ]);
            
            // Log first few announcements for debugging
            foreach ($allAnnouncements->take(3) as $ann) {
                Log::debug('Sample announcement', [
                    'id' => $ann->id,
                    'title' => $ann->title,
                    'is_published' => $ann->is_published,
                    'start_date' => $ann->start_date?->toDateTimeString(),
                    'end_date' => $ann->end_date?->toDateTimeString(),
                    'target_roles' => $ann->target_roles,
                    'target_units' => $ann->target_units,
                    'target_buildings' => $ann->target_buildings,
                ]);
            }
        }

        // Paginate manually
        $page = (int) $request->get('page', 1);
        $perPage = 20;
        $total = $filteredAnnouncements->count();
        $items = $filteredAnnouncements->slice(($page - 1) * $perPage, $perPage)->values();
        
        $announcements = new \Illuminate\Pagination\LengthAwarePaginator(
            $items,
            $total,
            $perPage,
            $page,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return AnnouncementResource::collection($announcements);
    }

    /**
     * Get announcement details and record view.
     */
    public function show(Announcement $announcement)
    {
        $user = Auth::user();

        if ($announcement->society_id !== $user->society_id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Record view if not already viewed
        AnnouncementView::firstOrCreate([
            'announcement_id' => $announcement->id,
            'user_id' => $user->id,
        ], [
            'viewed_at' => now(),
        ]);

        // Increment view count (optional, can be done via observer or job)
        $announcement->increment('views_count');

        return new AnnouncementResource($announcement);
    }

    /**
     * Acknowledge an announcement.
     */
    public function acknowledge(Request $request, Announcement $announcement)
    {
        $user = Auth::user();

        if ($announcement->society_id !== $user->society_id) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        if (! $announcement->require_acknowledgement) {
            return response()->json(['message' => 'Acknowledgement not required'], 400);
        }

        AnnouncementAcknowledgement::firstOrCreate([
            'announcement_id' => $announcement->id,
            'user_id' => $user->id,
        ], [
            'acknowledged_at' => now(),
        ]);

        $announcement->increment('acknowledged_count');

        return response()->json(['message' => 'Announcement acknowledged']);
    }
}
