<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Resources\AdResource;
use App\Models\AdSlot;
use App\Models\Advertisement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class AdController extends Controller
{
    /**
     * List available slots and pricing.
     */
    public function slots(Request $request)
    {
        $user = Auth::user();
        $slots = AdSlot::where('society_id', $user->society_id)
            ->where('status', true)
            ->get();
            
        return response()->json([
            'data' => $slots->map(function($slot) {
                return [
                    'id' => $slot->id,
                    'position' => $slot->position,
                    'name' => $slot->name,
                    'price_per_day' => $slot->price_per_day,
                    'is_available' => true, // Could check max_concurrent_ads logic here if complex
                ];
            })
        ]);
    }

    /**
     * Submit new ad request.
     */
    public function create(Request $request)
    {
        $request->validate([
             'slot_id' => 'required|exists:ad_slots,id',
             'title' => 'required|string|max:255',
             'media' => 'nullable|file|mimes:jpeg,png,jpg,gif,mp4,mov|max:20480', // 20MB max
             'image' => 'nullable|file|mimes:jpeg,png,jpg,gif,mp4,mov|max:20480', // Alias
             'start_date' => 'nullable|date|after_or_equal:today',
             'end_date' => 'nullable|date|after:start_date',
             'duration_days' => 'nullable|integer|min:1',
             'payment_reference' => 'nullable|string',
             'target_link' => 'nullable|url',
             'description' => 'nullable|string',
        ]);

        $file = $request->file('media') ?? $request->file('image');
        if (!$file) {
             return response()->json(['message' => 'Image or media file is required.', 'errors' => ['image' => ['Required']]], 422);
        }
        
        $user = Auth::user();
        $slot = AdSlot::where('id', $request->slot_id)->where('society_id', $user->society_id)->firstOrFail();
        
        // Calculate Dates
        if ($request->duration_days) {
             $start = $request->start_date ? Carbon::parse($request->start_date) : Carbon::today();
             $end = $start->copy()->addDays($request->duration_days - 1);
        } else {
             if (!$request->start_date || !$request->end_date) {
                  return response()->json(['message' => 'Either duration_days or start_date/end_date is required.'], 422);
             }
             $start = Carbon::parse($request->start_date);
             $end = Carbon::parse($request->end_date);
        }

        $days = $start->diffInDays($end) + 1; // Inclusive
        $days = max(1, $days); // Ensure at least 1 day
        $price = $days * $slot->price_per_day;

        // Upload Media
        $path = $file->store('ads', 'public');
        $mime = $file->getMimeType();
        $type = str_starts_with($mime, 'video') ? 'video' : 'image';
        
        // Create Ad
        $ad = Advertisement::create([
             'society_id' => $user->society_id,
             'user_id' => $user->id,
             'slot_id' => $slot->id,
             'title' => $request->title,
             'description' => $request->description,
             'media_url' => asset('storage/' . $path),
             'media_type' => $type,
             'target_link' => $request->target_link,
             'start_date' => $start,
             'end_date' => $end,
             'price' => $price,
             'payment_reference' => $request->payment_reference ?? 'PAY-' . strtoupper(uniqid()),
             'status' => 'pending_approval',
        ]);

        return new AdResource($ad);
    }

    /**
     * List user's ad requests and status.
     */
    public function myAds(Request $request)
    {
        $user = Auth::user();
        $ads = Advertisement::where('user_id', $user->id)
            ->with('slot')
            ->orderBy('created_at', 'desc')
            ->paginate(20);
            
        return AdResource::collection($ads);
    }

    /**
     * Fetch active ads for the current user/society.
     */
    public function active(Request $request)
    {
        $user = Auth::user();
        $today = now();
        
        // Logic: 
        // 1. Filter by society_id and status='active'
        // 2. Filter start_date <= now <= end_date
        
        $ads = Advertisement::where('society_id', $user->society_id)
            ->where('status', 'active')
            ->whereDate('start_date', '<=', $today)
            ->whereDate('end_date', '>=', $today)
            ->with('slot')
            ->get();
        
        // Group by position
        $grouped = $ads->groupBy(function($ad) {
            return $ad->slot ? $ad->slot->position : 'unknown';
        });
        
        $response = [
            'popup' => null,
            'community_top' => null,
            'community_bottom' => null,
        ];
        
        // Rotation Logic: Pick random one for each slot
        if (isset($grouped['app_open_popup']) && $grouped['app_open_popup']->count() > 0) {
            $response['popup'] = new AdResource($grouped['app_open_popup']->random());
        }
        if (isset($grouped['community_top']) && $grouped['community_top']->count() > 0) {
            $response['community_top'] = new AdResource($grouped['community_top']->random());
        }
        if (isset($grouped['community_bottom']) && $grouped['community_bottom']->count() > 0) {
            $response['community_bottom'] = new AdResource($grouped['community_bottom']->random());
        }
        
        return response()->json($response);
    }
}
