<?php

namespace App\Http\Controllers\Accountant;

use App\Http\Controllers\Controller;
use App\Models\BankAccount;
use App\Models\Bill;
use App\Models\Expense;
use App\Models\Payment;
use App\Models\Society;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use Inertia\Response;

class DashboardController extends Controller
{
    public function index(Request $request, Society $society): Response
    {
        $user = Auth::user();

        // Ensure user is accountant
        if (! $user->hasRole('accountant')) {
            abort(403, 'Only Accountant can access this dashboard.');
        }

        // Ensure user belongs to this society (defense in depth - middleware also checks this)
        if ($user->society_id !== $society->id) {
            abort(403, 'Unauthorized access to this society.');
        }

        // KPI 1: Total Collections (MTD - Month to Date)
        $totalCollections = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->sum('amount');

        // KPI 2: Outstanding Dues
        $outstandingDues = Bill::where('society_id', $society->id)
            ->whereIn('status', ['due', 'overdue'])
            ->sum('amount');

        // KPI 3: Month Expenses
        $monthExpenses = Expense::where('society_id', $society->id)
            ->whereMonth('expense_date', now()->month)
            ->whereYear('expense_date', now()->year)
            ->sum('amount');

        // KPI 4: Collection Rate
        $totalBills = Bill::where('society_id', $society->id)
            ->whereIn('status', ['due', 'paid', 'overdue'])
            ->sum('amount');
        $paidBills = Bill::where('society_id', $society->id)
            ->where('status', 'paid')
            ->sum('amount');
        $collectionRate = $totalBills > 0 ? round(($paidBills / $totalBills) * 100, 2) : 0;

        // KPI 5: Bank Balance (Total across all accounts)
        $bankBalance = BankAccount::where('society_id', $society->id)
            ->sum('current_balance');

        // KPI 6: Pending Reconciliation (Payments not reconciled)
        $pendingReconciliation = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->where('is_reconciled', false)
            ->count();

        // Recent Activity Feed (Financial only)
        $recentPayments = Payment::where('society_id', $society->id)
            ->with(['user:id,name', 'bill:id,bill_no'])
            ->latest()
            ->limit(5)
            ->get();

        $recentBills = Bill::where('society_id', $society->id)
            ->with(['user:id,name,unit_id', 'user.unit:id,unit_no'])
            ->latest()
            ->limit(5)
            ->get();

        $recentExpenses = Expense::where('society_id', $society->id)
            ->with(['createdBy:id,name'])
            ->latest()
            ->limit(5)
            ->get();

        // Monthly Collections Chart Data (Last 6 months)
        $monthlyCollections = Payment::where('society_id', $society->id)
            ->where('status', 'completed')
            ->where('created_at', '>=', now()->subMonths(6))
            ->get()
            ->groupBy(function ($payment) {
                return $payment->created_at->format('Y-m');
            })
            ->map(function ($payments, $month) {
                return [
                    'month' => $month,
                    'total' => (float) $payments->sum('amount'),
                ];
            })
            ->sortBy('month')
            ->values();

        // Monthly Expenses Chart Data (Last 6 months)
        $monthlyExpenses = Expense::where('society_id', $society->id)
            ->where('expense_date', '>=', now()->subMonths(6)->startOfMonth())
            ->get()
            ->groupBy(function ($expense) {
                return $expense->expense_date->format('Y-m');
            })
            ->map(function ($expenses, $month) {
                return [
                    'month' => $month,
                    'total' => (float) $expenses->sum('amount'),
                ];
            })
            ->sortBy('month')
            ->values();

        return Inertia::render('Accountant/Dashboard', [
            'society' => $society->only(['id', 'name', 'code']),
            'kpis' => [
                'total_collections_mtd' => (float) $totalCollections,
                'outstanding_dues' => (float) $outstandingDues,
                'month_expenses' => (float) $monthExpenses,
                'collection_rate' => $collectionRate,
                'bank_balance' => (float) $bankBalance,
                'pending_reconciliation' => $pendingReconciliation,
            ],
            'recentActivity' => [
                'payments' => $recentPayments->map(function ($payment) {
                    return [
                        'id' => $payment->id,
                        'payment_no' => $payment->payment_no,
                        'amount' => (float) $payment->amount,
                        'user_name' => $payment->user->name ?? 'N/A',
                        'bill_no' => $payment->bill->bill_no ?? null,
                        'is_reconciled' => $payment->is_reconciled,
                        'created_at' => $payment->created_at->format('Y-m-d H:i'),
                    ];
                }),
                'bills' => $recentBills->map(function ($bill) {
                    return [
                        'id' => $bill->id,
                        'bill_no' => $bill->bill_no,
                        'amount' => (float) $bill->amount,
                        'status' => $bill->status,
                        'user_name' => $bill->user->name ?? 'N/A',
                        'unit_no' => $bill->user->unit->unit_no ?? 'N/A',
                        'created_at' => $bill->created_at->format('Y-m-d H:i'),
                    ];
                }),
                'expenses' => $recentExpenses->map(function ($expense) {
                    return [
                        'id' => $expense->id,
                        'expense_no' => $expense->expense_no,
                        'amount' => (float) $expense->amount,
                        'category' => $expense->category,
                        'created_by_name' => $expense->createdBy->name ?? 'N/A',
                        'created_at' => $expense->created_at->format('Y-m-d H:i'),
                    ];
                }),
            ],
            'charts' => [
                'monthly_collections' => $monthlyCollections,
                'monthly_expenses' => $monthlyExpenses,
            ],
        ]);
    }
}
