<?php

namespace App\Helpers;

use App\Models\Role;
use App\Models\User;

class MenuHelper
{
    /**
     * Get menu items for a specific role.
     */
    public static function getMenuItemsForRole(string $roleName): array
    {
        return match ($roleName) {
            'society_admin' => self::getSocietyAdminMenu(),
            'sub_admin' => self::getSubAdminMenu(),
            'accountant' => self::getAccountantMenu(),
            default => [],
        };
    }

    /**
     * Get menu items for authenticated user based on permissions.
     */
    public static function getMenuItemsForUser(User $user): array
    {
        if (! $user->role) {
            return [];
        }

        $allMenuItems = self::getSocietyAdminMenu();
        $filteredMenuItems = [];

        foreach ($allMenuItems as $menuItem) {
            // Check if user has permission for this menu item
            if (isset($menuItem['permission'])) {
                if ($user->hasPermission($menuItem['permission'])) {
                    $filteredMenuItems[] = $menuItem;
                }
            } else {
                // If no permission specified, include it
                $filteredMenuItems[] = $menuItem;
            }
        }

        return $filteredMenuItems;
    }

    /**
     * Society Admin Menu (Full Access - 12 items)
     */
    private static function getSocietyAdminMenu(): array
    {
        return [
            [
                'title' => 'Dashboard',
                'href' => '/societies/{society}/dashboard',
                'icon' => 'LayoutGrid',
                'permission' => 'dashboard.view',
            ],
            [
                'title' => 'Society Settings',
                'href' => '/societies/{society}/settings',
                'icon' => 'Settings',
                'permission' => 'society.settings.view',
            ],
            [
                'title' => 'Residents',
                'href' => '/societies/{society}/residents',
                'icon' => 'Users',
                'permission' => 'residents.view',
            ],
            [
                'title' => 'Units & Buildings',
                'href' => '/societies/{society}/units',
                'icon' => 'Building2',
                'permission' => 'units.view',
            ],
            [
                'title' => 'Billing',
                'href' => '/societies/{society}/billing',
                'icon' => 'Receipt',
                'permission' => 'billing.bills.view',
            ],
            [
                'title' => 'Payments',
                'href' => '/societies/{society}/payments',
                'icon' => 'CreditCard',
                'permission' => 'payments.view',
            ],
            [
                'title' => 'Bank Accounts',
                'href' => '/societies/{society}/bank-accounts',
                'icon' => 'Landmark',
                'permission' => 'bank_accounts.view',
            ],
            [
                'title' => 'Expenses',
                'href' => '/societies/{society}/expenses',
                'icon' => 'TrendingDown',
                'permission' => 'expenses.view',
            ],
            [
                'title' => 'Maintenance',
                'href' => '/societies/{society}/maintenance',
                'icon' => 'Wrench',
                'permission' => 'maintenance.view',
            ],
            [
                'title' => 'Visitors',
                'href' => '/societies/{society}/visitors',
                'icon' => 'ShieldCheck',
                'permission' => 'visitors.view',
            ],
            [
                'title' => 'Announcements',
                'href' => '/societies/{society}/announcements',
                'icon' => 'Megaphone',
                'permission' => 'announcements.view',
            ],
            [
                'title' => 'Reports',
                'href' => '/societies/{society}/reports',
                'icon' => 'FileText',
                'permission' => 'reports.view',
            ],
            [
                'title' => 'Directory',
                'href' => '/societies/{society}/directory',
                'icon' => 'BookOpen',
                'permission' => 'directory.view',
            ],
        ];
    }

    /**
     * Sub-Admin Menu (Limited Access - 8 items)
     */
    private static function getSubAdminMenu(): array
    {
        return [
            [
                'title' => 'Dashboard',
                'href' => '/societies/{society}/sub-admin/dashboard',
                'icon' => 'LayoutGrid',
                'permission' => 'dashboard.view',
            ],
            [
                'title' => 'Residents',
                'href' => '/societies/{society}/sub-admin/residents',
                'icon' => 'Users',
                'permission' => 'residents.view',
            ],
            [
                'title' => 'Units',
                'href' => '/societies/{society}/sub-admin/units',
                'icon' => 'Building2',
                'permission' => 'units.view',
            ],
            [
                'title' => 'Billing',
                'href' => '/societies/{society}/sub-admin/billing',
                'icon' => 'Receipt',
                'permission' => 'billing.bills.view',
            ],
            [
                'title' => 'Expenses',
                'href' => '/societies/{society}/sub-admin/expenses',
                'icon' => 'TrendingDown',
                'permission' => 'expenses.view',
            ],
            [
                'title' => 'Maintenance',
                'href' => '/societies/{society}/sub-admin/maintenance',
                'icon' => 'Wrench',
                'permission' => 'maintenance.view',
            ],
            [
                'title' => 'Visitors',
                'href' => '/societies/{society}/sub-admin/visitors',
                'icon' => 'ShieldCheck',
                'permission' => 'visitors.view',
            ],
            [
                'title' => 'Announcements',
                'href' => '/societies/{society}/sub-admin/announcements',
                'icon' => 'Megaphone',
                'permission' => 'announcements.view',
            ],
        ];
    }

    /**
     * Accountant Menu (Financial Only - 6 items)
     */
    private static function getAccountantMenu(): array
    {
        return [
            [
                'title' => 'Financial Dashboard',
                'href' => '/societies/{society}/accountant/dashboard',
                'icon' => 'LayoutGrid',
                'permission' => 'dashboard.view',
            ],
            [
                'title' => 'Billing',
                'href' => '/societies/{society}/accountant/bills',
                'icon' => 'Receipt',
                'permission' => 'billing.bills.view',
            ],
            [
                'title' => 'Payments & Reconciliation',
                'href' => '/societies/{society}/accountant/payments',
                'icon' => 'CreditCard',
                'permission' => 'payments.view',
            ],
            [
                'title' => 'Bank Accounts',
                'href' => '/societies/{society}/accountant/bank-accounts',
                'icon' => 'Landmark',
                'permission' => 'bank_accounts.view',
            ],
            [
                'title' => 'Expenses',
                'href' => '/societies/{society}/accountant/expenses',
                'icon' => 'TrendingDown',
                'permission' => 'expenses.view',
            ],
            [
                'title' => 'Financial Reports',
                'href' => '/societies/{society}/accountant/reports',
                'icon' => 'FileText',
                'permission' => 'reports.financial',
            ],
        ];
    }
}
