<?php

namespace App\Console\Commands;

use App\Models\CollectorWallet;
use App\Models\Payment;
use Illuminate\Console\Command;

class FixCollectorWallets extends Command
{
    protected $signature = 'fix:collector-wallets';
    protected $description = 'Credit collector wallets for existing cash payments that were not credited';

    public function handle()
    {
        $this->info('Finding collector cash payments without wallet credits...');

        $payments = Payment::where('payment_method', 'cash')
            ->where('collector_role', 'collector')
            ->where('status', 'completed')
            ->whereNotNull('collected_by')
            ->get();

        $this->info("Found {$payments->count()} collector cash payments");

        $credited = 0;
        foreach ($payments as $payment) {
            $wallet = CollectorWallet::getOrCreate($payment->collected_by, $payment->society_id);

            // Check if transaction already exists
            $exists = $wallet->transactions()
                ->where('reference_id', $payment->id)
                ->where('reference_type', Payment::class)
                ->exists();

            if (!$exists) {
                $wallet->credit($payment->amount, $payment->id, 'Payment #' . $payment->payment_no);
                $this->info("Credited ₹{$payment->amount} for {$payment->payment_no}");
                $credited++;
            } else {
                $this->line("Already credited: {$payment->payment_no}");
            }
        }

        $this->info("Done! Credited {$credited} payments.");

        // Show final wallet balances
        $wallets = CollectorWallet::with('user:id,name')->get();
        foreach ($wallets as $wallet) {
            $this->info("Wallet for {$wallet->user->name}: ₹{$wallet->balance}");
        }

        return 0;
    }
}
