<?php

namespace App\Console\Commands;

use App\Models\Society;
use App\Models\Subscription;
use App\Services\EmailService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CheckSubscriptionExpiry extends Command
{
    /**
     * The name and signature of the console command.
     */
    protected $signature = 'subscriptions:check-expiry';

    /**
     * The console command description.
     */
    protected $description = 'Check for expiring and expired subscriptions and send email notifications';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('Checking subscription expiry...');

        $now = now();

        // 1. Find subscriptions expiring in 7 days
        $expiringIn7Days = Subscription::where('status', 'active')
            ->whereDate('end_date', $now->copy()->addDays(7)->toDateString())
            ->with('society')
            ->get();

        foreach ($expiringIn7Days as $subscription) {
            if ($subscription->society) {
                EmailService::sendExpiringEmail($subscription->society, $subscription, 7);
                $this->info("Sent 7-day expiry notice to: {$subscription->society->name}");
            }
        }

        // 2. Find subscriptions expiring in 3 days
        $expiringIn3Days = Subscription::where('status', 'active')
            ->whereDate('end_date', $now->copy()->addDays(3)->toDateString())
            ->with('society')
            ->get();

        foreach ($expiringIn3Days as $subscription) {
            if ($subscription->society) {
                EmailService::sendExpiringEmail($subscription->society, $subscription, 3);
                $this->info("Sent 3-day expiry notice to: {$subscription->society->name}");
            }
        }

        // 3. Find subscriptions expiring tomorrow
        $expiringTomorrow = Subscription::where('status', 'active')
            ->whereDate('end_date', $now->copy()->addDay()->toDateString())
            ->with('society')
            ->get();

        foreach ($expiringTomorrow as $subscription) {
            if ($subscription->society) {
                EmailService::sendExpiringEmail($subscription->society, $subscription, 1);
                $this->info("Sent 1-day expiry notice to: {$subscription->society->name}");
            }
        }

        // 4. Find subscriptions that just expired (mark as expired and notify)
        $justExpired = Subscription::where('status', 'active')
            ->whereDate('end_date', '<', $now->toDateString())
            ->with('society')
            ->get();

        foreach ($justExpired as $subscription) {
            $subscription->update(['status' => 'expired']);
            
            if ($subscription->society) {
                EmailService::sendExpiredEmail($subscription->society, $subscription);
                $this->info("Marked as expired and notified: {$subscription->society->name}");
            }
        }

        // Subscription expiry check completed
        // Use command output for monitoring instead of logs

        $this->info('Subscription expiry check completed.');
        return Command::SUCCESS;
    }
}
