<?php

namespace App\Console\Commands;

use App\Services\ResidentEmailService;
use Illuminate\Console\Command;

class CheckOverdueBills extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:check-overdue-bills';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check for overdue bills, apply interest, and send notifications';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $today = now()->format('Y-m-d');

        // 1. Send due date reminders (7 days, 3 days, 1 day before)
        $this->sendDueReminders($today);

        // 2. Mark bills as overdue if due_date < today
        $billsToMarkOverdue = \App\Models\Bill::whereIn('status', ['due', 'partial'])
            ->where('due_date', '<', $today)
            ->get();

        foreach ($billsToMarkOverdue as $bill) {
            $bill->update(['status' => 'overdue']);
            
            // Send overdue notification email
            $daysOverdue = now()->diffInDays($bill->due_date);
            ResidentEmailService::sendBillOverdueEmail($bill, $daysOverdue);
        }

        // 3. Apply Interest to Overdue Bills
        $overdueBills = \App\Models\Bill::whereIn('status', ['overdue'])
            ->where('due_date', '<', $today)
            ->where(function ($query) use ($today) {
                $query->whereNull('last_interest_applied_at')
                      ->orWhere('last_interest_applied_at', '<', $today);
            })
            ->with(['society.settings'])
            ->get();

        $count = 0;
        
        foreach ($overdueBills as $bill) {
            $settings = $bill->society->settings ?? null;
            if (! $settings) continue;

            $calcType = $settings->late_fee_calculation_type ?? 'none';
            if ($calcType === 'none') continue;

            $interestAmount = 0;
            $pendingAmount = $bill->amount;

            if ($calcType === 'fixed_daily') {
                $interestAmount = $settings->late_fee_value;
            } elseif ($calcType === 'percentage_daily') {
                $interestAmount = ($pendingAmount * $settings->late_fee_value) / 100;
            }

            if ($interestAmount > 0) {
                $bill->late_fee = ($bill->late_fee ?? 0) + $interestAmount;
                $bill->last_interest_applied_at = $today;
                $bill->save();
                $count++;
            }
        }
        
        $this->info("Processed overdue bills. Interest applied to {$count} bills.");
    }

    /**
     * Send due date reminders for bills due in 7, 3, and 1 days.
     */
    protected function sendDueReminders(string $today): void
    {
        $reminderDays = [7, 3, 1];

        foreach ($reminderDays as $days) {
            $dueDate = now()->addDays($days)->format('Y-m-d');
            
            $bills = \App\Models\Bill::whereIn('status', ['due', 'partial', 'approved'])
                ->whereDate('due_date', $dueDate)
                ->with(['user', 'society.settings'])
                ->get();

            foreach ($bills as $bill) {
                if ($bill->user && !empty($bill->user->email)) {
                    ResidentEmailService::sendBillDueReminderEmail($bill, $days);
                }
            }

            $this->info("Sent {$bills->count()} due reminders for bills due in {$days} days.");
        }
    }
}

