<?php

namespace App\Console\Commands;

use App\Models\Bill;
use App\Models\BillCycle;
use App\Models\MaintenanceStructure;
use App\Models\Role;
use App\Models\Society;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class AutoGenerateBillingCycle extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'billing:auto-generate-cycle';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Automatically create billing cycle for the ended month and generate bills on the 1st of every month';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting automatic billing cycle generation...');

        // Get the previous month
        $previousMonth = Carbon::now()->subMonth();
        $year = $previousMonth->year;
        $month = $previousMonth->month;
        $monthName = $previousMonth->format('F Y');

        $this->info("Processing billing cycle for: {$monthName}");

        // Get all active societies
        $societies = Society::where('status', 'active')->get();

        if ($societies->isEmpty()) {
            $this->warn('No active societies found.');
            return 0;
        }

        $totalCyclesCreated = 0;
        $totalBillsGenerated = 0;

        foreach ($societies as $society) {
            try {
                // Check if bill cycle already exists for this month
                $existingCycle = BillCycle::where('society_id', $society->id)
                    ->where('year', $year)
                    ->whereMonth('generate_date', $month)
                    ->first();

                if ($existingCycle) {
                    $this->warn("Bill cycle already exists for {$society->name} - {$monthName}. Skipping...");
                    continue;
                }

                // Calculate dates
                $generateDate = $previousMonth->copy()->startOfMonth();
                $lastDayOfMonth = $previousMonth->copy()->endOfMonth();
                $graceDays = 5; // 5 days grace period before end of month
                $dueDate = $lastDayOfMonth->copy()->subDays($graceDays);

                // Create bill cycle
                $billCycle = BillCycle::create([
                    'society_id' => $society->id,
                    'name' => "Billing Cycle - {$monthName}",
                    'year' => $year,
                    'generate_date' => $generateDate->format('Y-m-d'),
                    'due_date' => $dueDate->format('Y-m-d'),
                    'status' => 'draft',
                ]);

                $this->info("Created bill cycle for {$society->name}");

                // Generate bills for this cycle
                $billsCount = $this->generateBillsForCycle($society, $billCycle, $dueDate);

                $totalCyclesCreated++;
                $totalBillsGenerated += $billsCount;

                $this->info("Generated {$billsCount} bills for {$society->name}");

                Log::info("Auto-generated billing cycle", [
                    'society_id' => $society->id,
                    'society_name' => $society->name,
                    'bill_cycle_id' => $billCycle->id,
                    'month' => $monthName,
                    'bills_count' => $billsCount,
                ]);

            } catch (\Exception $e) {
                $this->error("Error processing society {$society->name}: " . $e->getMessage());
                Log::error("Failed to auto-generate billing cycle for society", [
                    'society_id' => $society->id,
                    'society_name' => $society->name,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString(),
                ]);
                continue;
            }
        }

        $this->info("✅ Completed! Created {$totalCyclesCreated} billing cycles with {$totalBillsGenerated} total bills.");
        return 0;
    }

    /**
     * Generate bills for a bill cycle using default maintenance structure.
     */
    protected function generateBillsForCycle(Society $society, BillCycle $billCycle, Carbon $dueDate): int
    {
        $residentRole = Role::where('name', 'resident')->first();

        if (!$residentRole) {
            $this->warn("Resident role not found for society {$society->name}");
            return 0;
        }

        // Get all active residents with units (only owners receive bills)
        $residents = User::where('society_id', $society->id)
            ->where('role_id', $residentRole->id)
            ->where('status', 'active')
            ->whereNotNull('unit_id')
            ->where(function ($query) {
                // Only include owners (exclude family_member and tenant)
                $query->where('type', 'owner')
                      ->orWhereNull('type'); // Also include null type for backward compatibility
            })
            ->with(['unit.maintenanceStructure'])
            ->get();

        if ($residents->isEmpty()) {
            $this->warn("No active residents with units found for society {$society->name}");
            return 0;
        }

        // Get default maintenance structure for the society
        $defaultStructure = MaintenanceStructure::where('society_id', $society->id)
            ->orderBy('id')
            ->first();

        $billsCreated = 0;

        DB::transaction(function () use ($society, $billCycle, $residents, $defaultStructure, $dueDate, &$billsCreated) {
            // Get the next bill number for this society
            $prefix = 'BILL-SOC-' . date('Y') . '-';
            
            $lastBill = Bill::where('society_id', $society->id)
                ->where('bill_no', 'like', $prefix . '%')
                ->orderBy('id', 'desc')
                ->first();

            $sequence = 0;
            if ($lastBill) {
                $parts = explode('-', $lastBill->bill_no);
                $lastSeq = end($parts);
                if (is_numeric($lastSeq)) {
                    $sequence = intval($lastSeq);
                }
            }

            // Start sequence for this batch
            $currentSequence = $sequence + 1;

            foreach ($residents as $resident) {
                // Determine Structure to use
                // Priority 1: Unit's Assigned Structure
                // Priority 2: Default Structure for Society
                // Priority 3: Fallback to 0 (only if no structures exist)
                
                $structureToUse = null;
                if ($resident->unit && $resident->unit->maintenanceStructure) {
                    $structureToUse = $resident->unit->maintenanceStructure;
                }
                if (!$structureToUse && $defaultStructure) {
                    $structureToUse = $defaultStructure;
                }

                // Calculate amounts using structure or fallback to 0
                $maintenance = $structureToUse ? $structureToUse->maintenance : 0;
                $parking = $structureToUse ? $structureToUse->parking : 0;
                $otherCharges = $structureToUse ? $structureToUse->other_charges : 0;
                
                $water = 0; // Variable
                $electricity = 0; // Variable
                $gst = 0;
                $discount = 0;

                // Check if bill already exists for this cycle and resident
                $existingBill = Bill::where('bill_cycle_id', $billCycle->id)
                    ->where('user_id', $resident->id)
                    ->first();

                if ($existingBill) {
                    continue; // Skip if bill already exists
                }

                // Generate standardized bill number
                $billNo = $prefix . str_pad($currentSequence++, 6, '0', STR_PAD_LEFT);

                $amount = $maintenance + $water + $electricity + $parking + $otherCharges + $gst - $discount;

                // Determine initial status
                // If amount is 0, set to 'draft' (not paid, needs review)
                // Otherwise, set to 'under_review' (will be approved when admin posts)
                $initialStatus = ($amount == 0) ? 'draft' : 'under_review';

                Bill::create([
                    'society_id' => $society->id,
                    'bill_cycle_id' => $billCycle->id,
                    'user_id' => $resident->id,
                    'bill_no' => $billNo,
                    'amount' => $amount,
                    'maintenance' => $maintenance,
                    'water' => $water,
                    'electricity' => $electricity,
                    'parking' => $parking,
                    'other_charges' => $otherCharges,
                    'gst' => $gst,
                    'discount' => $discount,
                    'status' => $initialStatus,
                    'due_date' => $dueDate->format('Y-m-d'),
                    'generated_by' => null, // System generated
                ]);

                $billsCreated++;
            }

            // Update bill cycle status to 'generated'
            $billCycle->update(['status' => 'generated']);
        });

        return $billsCreated;
    }
}
